<?php

use Advertikon\Element\Anchor;
use Advertikon\Element\Bootstrap\Button;
use Advertikon\Element\Bootstrap\Color;
use Advertikon\Element\Bootstrap\FormGroup;
use advertikon\element\Bootstrap\Message;
use Advertikon\Element\FancyCheckBox;
use Advertikon\Element\Select;
use Advertikon\Element\Slider;
use advertikon\SessionMessage;
use Advertikon\Setting;
use Advertikon\Stripe\Advertikon;
use Advertikon\Stripe\PaymentOption;
use Advertikon\Stripe\StripeAccount;
use Stripe\Stripe;

/**
 * Advertikon Stripe admin model
 * @author Advertikon
 * @package Stripe
 * @version 5.0.44    
 */

/**
 * Class ModelExtensionPaymentAdvertikonStripe
 * @property $request Request
 */
class ModelExtensionPaymentAdvertikonStripe extends Model {
    /** @var Advertikon */
	protected $a = null;

	/**
	 * ModelExtensionPaymentAdvertikonStripe constructor.
	 * @param $registry
	 * @throws \Advertikon\Exception
	 */
	public function __construct( $registry ) {
		parent::__construct( $registry );
		$this->a = Advertikon::instance( $registry );
	}

	/**
	 * Creates all the extension's tables
	 * @return void
	 */
	public function create_tables() {
		$this->db->query(
			"CREATE TABLE IF NOT EXISTS `" . DB_PREFIX . $this->a->tables['order'] . "`
			(
				`id`    INT(11) UNSIGNED AUTO_INCREMENT,
				`data`  TEXT,
				`code`  VARCHAR(50),
				INDEX(`code`),
				PRIMARY KEY(`id`)
			)
			ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_bin"
		);

		$this->db->query(
			"CREATE TABLE IF NOT EXISTS `" . DB_PREFIX . $this->a->tables['recurring_table'] . "`
			(
				`id`                 INT(11)       UNSIGNED AUTO_INCREMENT,
				`recurring_order_id` INT           UNSIGNED,
				`subscription_id`    VARCHAR( 100 ),
				`next`               TEXT,
				`total_tax`          TEXT,
				`shipping_tax`       TEXT,
				`invoices`           TEXT,
				UNIQUE(`recurring_order_id`),
				PRIMARY KEY(`id`)
			)
			ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_bin"
		); 

		$this->db->query(
			"CREATE TABLE IF NOT EXISTS `" . DB_PREFIX . $this->a->tables['customer_table'] . "`
			(
				`id`             INT(11)      UNSIGNED AUTO_INCREMENT,
				`oc_customer_id` INT(11)      UNSIGNED,
				`stripe_id`      VARCHAR(255),
				`date_added`     DATETIME,
				`date_modified`  TIMESTAMP,
				`description`    VARCHAR(255),
				`is_test`        TINYINT(1),
				`metadata`       TEXT,
				PRIMARY KEY(`id`),
				INDEX(`oc_customer_id`,`stripe_id`)
			)
			ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_bin"
		);
	}

    /**
     * Validates configuration data
     * @param array $errors
     * @return array
     * @throws \Advertikon\Exception
     */
	public function validate_configs() {
		$errors = [];

		try {
			// Check permissions to modify the extension's settings
			if ( !$this->a->has_permissions( 'modify' ) ) {
				throw new \Advertikon\Exception( $this->a->__( 'You have no permissions to modify extension settings' ) );
			}

			// Strip whitespace from Stripe API keys
			foreach( $this->a->post( 'account', array() ) as $name => $account ) {
				foreach( $account as $key => $val ) {
					if ( in_array( $key, array(
							'test_public_key',
							'test_secret_key',
							'live_public_key',
							'live_secret_key',
							'live_signing_key',
							'test_signing_key'
						) ) ) {
						$this->request->post[ Setting::prefix_name( 'account', $this->a ) ][ $name ][ $key ] = trim( $val );
					}
				}
			}

			// Geo zone
			if ( !$this->a->post( 'geo_zone' ) ) {
					$errors['geo_zone'] = $this->a->__( 'At least one Geo-zone should be selected' );
			}

			// Store
			if ( !$this->a->post( 'stores' ) ) {
					$errors['stores'] = $this->a->__( 'At least one store should be selected' );
			}

			// Statement descriptor check
			if( $sd = $this->a->post( 'statement_descriptor' ) ) {
				if( preg_match( '/[^0-9A-Za-z _-]/', $sd ) ) {
					$errors['statement_descriptor'] = $this->a->__( 'Statement descriptor may contain only alphanumeric symbols' );

				} elseif( strlen( $sd ) > 22 ) {
					$errors['statement_descriptor'] = $this->a->__( 'Statement descriptor may not be longer than 22 characters' );
				}
			}

			if ( !isset( $errors['account'] ) ) {
				$errors['account'] = array();
			}

			foreach( $this->a->post( 'account', array() )  as $name => $data ) {
				$name = $this->a->strip_prefix( $name );

				if ( !isset( $errors['account'][ $name ] ) ) {
					$errors['account'][ $name ] = array();
				}

				// Presence of test or live key pair depend on sandbox mode
				if ( $this->a->post( 'test_mode' ) ) {
					if ( empty( $data['test_secret_key'] ) )	{
						$errors['account'][ $name ]['test_secret_key'] = $this->a->__( 'Test Secret Key Required!' );
					}

					if ( empty( $data['test_public_key'] ) )	{
						$errors['account'][ $name ]['test_public_key'] = $this->a->__( 'Test Public Key Required!' );
					}

					if ( empty( $data['test_signing_key'] ) )	{
						$errors['account'][ $name ]['test_signing_key'] = $this->a->__( 'Test Webhook Signing Key Required!' );
					}

				} else {
					if ( empty( $data['live_secret_key'] ) )	{
						$errors['account'][ $name ]['live_secret_key'] = $this->a->__( 'Live Secret Key Required!' );
					}

					if ( empty( $data['live_public_key'] ) )	{
						$errors['account'][ $name ]['live_public_key'] = $this->a->__( 'Live Public Key Required!' );
					}

					if ( empty( $data['live_signing_key'] ) )	{
						$errors['account'][ $name ]['live_signing_key'] = $this->a->__( 'Live Webhook Signing Key Required!' );
					}
				}

				// Account name
				if ( empty( $data['account_name'] ) ) {
					$errors['account'][ $name ]['account_name'] = $this->a->__( 'Account name required!' );
				}

				if ( !$errors['account'][$name] ) {
					unset( $errors['account'][$name]);
				}
			}

			if ( !$errors['account'] ) {
				unset( $errors['account'] );
			}
			
			// Minimum total amount adjusting (if needed)
			$min_total = $this->a->post( 'total_min' ) ? $this->a->post( 'total_min' ) : 0;
			$correct_min_total = $this->a->check_min_amount( $min_total, $this->config->get( 'config_currency' ) );

			if ( true !== $correct_min_total ) {
				$this->request->post[ Setting::prefix_name( 'total_min', $this->a ) ] = $this->currency->format(
					$correct_min_total,
					$this->config->get( 'config_currency' ),
					1,
					false
				);

				SessionMessage::add( $this->a->__( 'Minimum total amount has been adjusted to meet Stripe requirements' ) );
			}

			$this->request->post[ Setting::prefix_name( 'total_max', $this->a ) ] = is_numeric( $this->a->post( 'total_max' ) ) ? $this->a->post( 'total_max' ) : null;

			if ( null !== $this->a->post( 'total_max' ) ) {
				if ( $this->a->post( 'total_max' ) < $this->a->post( 'total_min' ) ) {
					$errors['total_max'] = $this->a->__( 'Maximum total amount cannot be lesser then minimum total amount' );
				}
			}

			// Charge description
			if ( !$this->a->post( 'charge_description' ) ) {
				$errors['charge_description'] = $this->a->__( 'Specify description for charge' );
			}

			// Customer description
			if ( !$this->a->post( 'customer_description' ) ) {
				$errors['customer_description'] = $this->a->__( 'Specify customer description' );
			}

			// Done without errors
			if ( empty( $errors ) ) {
				unset( $_POST['template'] );

			//Got errors
			} else {
				throw new \Advertikon\Exception ( $this->a->__( 'In order to continue you have to correct some data' ) );
			}

		} catch( \Advertikon\Exception $e ) {
			SessionMessage::add( $e->getMessage(), Message::ERROR );
			$this->a->error( $errors );

			if( !$errors ) {
			    $errors[] = 'error';
            }
		} 

		return $errors;
	}

	/**
     * Selects label class for payment card verification field to show at admin area
     * @param string $status 
     * @return string
     */
    public function get_check_class( $status ) {
    	$ret = 'label';

    	switch ( $status ){
    		case 'pass' :
    			$ret .= ' label-success';
    			break;
    		case 'fail' :
    			$ret .= ' label-danger';
    			break;
    		case 'unavailable' :
    		case 'unchecked' :
    			$ret .= ' label-default';
    			break;
    	}

    	return $ret;
    }

    /**
     * @param array $data
     * @param array $resp
     * @return array
     * @throws \Advertikon\Exception
     * @throws Exception
     */
	public function get_controls( array $data, array $resp ) {
		/** @var \Advertikon\Stripe\Option $option */
		$option = $this->a->option();

		$data['add_account_btn'] = (new FormGroup())
			->label( new Anchor(
				'//dashboard.stripe.com/account/apikeys',
				$this->a->__( 'To obtain API keys visit Stripe Dashboard' )
			) )
			->element(
				(new Button())
					->id( 'add-account' )
					->getClass()->add( 'pull-right' )
					->icon( 'fa-plus' )
					->isPrimary()
					->title( $this->a->__( 'Add Stripe account' ) )
			);
		
		$name = 'payment_method';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Capture mode' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Authorize: payment needs to be accepted manually over Stripe account or admin panel, Capture: payments are accepted automatically. Authorize if fraud: if fraud is detected by one of the enabled anti-fraud extensions a payment will be authorized, otherwise - captured' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => $option->payment_option(),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ? $resp[ $name ] : null,
		) );
		
		$name = 'payment_currency';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Payment currency' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'When you have several Stripe accounts with different currencies (you can add them at "API Keys" tab) the extension selects a proper account depending on current storefront currency. E.g. customer selects GBP as storefront currency and you have two accounts USD (default) and GBP, then GBP account will be automatically selected to place payment in. If there is no corresponding account (e.g. customer selects CAD) then default Stipe account is selected. Charge currency depends on this setting. If the setting is "Store" then currency of default account will be selected (USD) and the exchange rate of OpenCart will be applied (CAD => USD). If the setting is set to "Order" then storefront currency will be selected (CAD) and the exchange rate of Stripe will be applied (CAD => USD)' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => array(
					Advertikon::CURRENCY_STORE => $this->a->__( 'Store' ),
					Advertikon::CURRENCY_ORDER => $this->a->__( 'Order' ),
				),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ? $resp[ $name ] : null,
		) );
		
		$name = 'charge_description';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Charge description' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Customizable charge description. Supported variables are:' ) .
				implode( ', ', $option->shortcode() ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name, 'charge for {customer_email}' ),
				'id'          => 'input-' . $name,
				'placeholder' => $this->a->__( 'Charge description' ),
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'customer_description';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Customer description' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Customer description to be displayed in Stripe Dashboard. Supported variables are:' ) .
				implode( ', ', $option->shortcode() ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name, '{customer_full_name}' ),
				'id'          => 'input-' . $name,
				'placeholder' => $this->a->__( 'Customer description' ),
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'statement_descriptor';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Statement descriptor' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'An arbitrary string to be displayed on your customer\'s credit card statement' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => $this->a->__( 'Statement descriptor' ),
				'custom_data' => [ 'maxlength' => 22 ],
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'total_min';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Min total amount' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Minimum order amount' ),
			'element'   => $this->a->r()->render_input_group( array(
				'element'      =>  array(
					'type'        => 'number',
					'name'        => $name,
					'class'       => 'form-control',
					'value'       => $this->a->get_value_from_post( $name ),
					'id'          => 'input-' . $name,
					'placeholder' => $this->a->__( 'Min total amount' ),
				),
				'error'       => isset( $resp[ $name ] ) ?
				$resp[ $name ] : null,
				'addon_before' => $this->currency->getSymbolLeft(
						$this->config->get( 'config_currency' )
						),
				'addon_after'  => $this->currency->getSymbolRight(
						$this->config->get( 'config_currency' )
						),
			) ),
		) );
		
		$name = 'total_max';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Max total amount' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Maximum order amount' ),
			'element'   => $this->a->r()->render_input_group( array(
				'element'      =>  array(
					'type'        => 'number',
					'name'        => $name,
					'class'       => 'form-control',
					'value'       => $this->a->get_value_from_post( $name ),
					'id'          => 'input-' . $name,
					'placeholder' => $this->a->__( 'Max total amount' ),
				),
				'error'       => isset( $resp[ $name ] ) ?
				$resp[ $name ] : null,
				'addon_before' => $this->currency->getSymbolLeft(
						$this->config->get( 'config_currency' )
						),
				'addon_after'  => $this->currency->getSymbolRight(
						$this->config->get( 'config_currency' )
						),
			) ),
		) );
		
		$name = 'geo_zone';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Geo Zones' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Permitted Geo Zones' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control select2',
				'active'      => $this->a->get_value_from_post( $name, array( -1 ) ),
				'value'       => array_replace( array( '-1' => $this->a->__( 'All Geo Zones') ), $option->geo_zone() ),
				'id'          => 'input-' . $name,
				'multiple'    => true,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
//		$id = 'avail_systems';
//		$name = $this->a->build_name( $id, '-' );
//		$data[ $id ] = $this->a->r()->render_form_group( array(
//			'label'     => $this->a->__( 'Payment systems' ),
//			'label_for' => $id,
//			'tooltip'   => $this->a->__( 'Permitted payment systems' ),
//			'element'   => array(
//				'type'        => 'select',
//				'name'        => $name,
//				'class'       => 'form-control select2',
//				'active'      => $this->a->get_value_from_post( $name, array( 0 ) ),
//				'value'       => array_merge( array( '0' => $this->a->__( 'All systems' ) ),
//                    PaymentOption::getAllNames() ),
//				'id'          => $id,
//				'multiple'    => true,
//			),
//			'error'       => isset( $resp[ $name ] ) ?
//			$resp[ $name ] : null,
//		) );
		
		$name = 'stores';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Stores' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Permitted stores' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control select2',
				'active'      => $this->a->get_value_from_post( $name, array( -1 ) ),
				'value'       => array_replace( array( '-1' => $this->a->__( 'All stores' ) ), $option->store() ),
				'id'          => 'input-' . $name,
				'multiple'    => true,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'customer_groups';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Customer groups' ),
			'label_for' => 'input-' . $name,
			'tootlip'   => $this->a->__( 'Permitted customer groups' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control select2',
				'active'      => $this->a->get_value_from_post( $name, array( 0 ) ),
				'value'       => array_merge( array( '0' => $this->a->__( 'All groups' ) ), $option->customer_group() ),
				'id'          => 'input-' . $name,
				'multiple'    => true,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'recurring_only';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Recurring only' ),
			'tooltip' => $this->a->__( 'Apply payment method only to orders with subscriptions' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'recurring_only_logged';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Recurring for registered only' ),
			'tooltip' => $this->a->__( 'Handle recurring payments only if customer is logged in' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );
		
		$name = 'sort_order';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Sort Order' ),
			'label_for' => 'input-' . $name,
			'element'   => $this->a->r( array(
				'type'        => 'number',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => $this->a->__( 'Sort Order' ),
			) ),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'receipt_email';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Receipt Email' ),
			'tooltip' => $this->a->__( 'Whether to send charge receipt at customer\'s email address. Receipts will not be sent for test mode charges. In live mode if set to "Yes" will override Stripe Dashboard\'s setting, if set to "No" Stripe Dashboard\'s setting will be applied' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );
		
		$name = 'notify_customer';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Notify customer' ),
			'tooltip' => $this->a->__( 'Whether to notify customer on payment capture or refund which made from Stripe Dashboard' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'          => $name,
				'id'            => $name,
				'value'         => $this->a->get_value_from_post( $name ),
				'text_on'       => $this->a->__( 'Yes' ),
				'text_off'      => $this->a->__( 'No' ),
				'dependent_off' => '#override',
			) ),
		) );
		
		$name = 'override';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Override' ),
			'tooltip' => $this->a->__( 'Whether to override order status, blocked by anti-fraud extensions, on payment capture or refund made from Stripe Dashboard' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'         => $name,
				'id'           => $name,
				'value'        => $this->a->get_value_from_post( $name ),
				'text_on'      => $this->a->__( 'Yes' ),
				'text_off'     => $this->a->__( 'No' ),
				'dependent_on' => '#notify_customer',
			) ),
		) );
		
		$name = 'hide_button';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Hide button' ),
			'tooltip' => $this->a->__( 'If you are using a quick (one page) checkout extension and for some reason there are two confirm buttons - enable this setting to hide one of them' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );
		
		$name = 'button_class';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Button\'s CSS class' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'If you are using a quick (one page) checkout extension and it uses a non-standard selectors to trigger payment button you can add such selectors - comma separated list of class names - to the payment button' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => $this->a->__( 'Button\'s CSS class' ),
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'status';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Extension status' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
			) ),
		) );
		
		$name = 'status_authorized';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Authorized Payment Status' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Orders status when payment was just authorized (customer\'s card has not been charged)' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => $option->order_status(),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'status_captured';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Captured Payment Status' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Which status to assign to an order upon payment capture' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => $option->order_status(),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'status_voided';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Refunded Payment Status' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Which status to assign to an order upon payment full refund' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => $option->order_status(),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
//		$name = 'show_systems';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Show permitted payment systems' ),
//			'tooltip' => $this->a->__( 'Defines whether to show icons of permitted payment systems, along with payment method name on checkout' ),
//			'element' => $this->a->r()->render_fancy_checkbox( array(
//				'name'     => $name,
//				'id'       => $name,
//				'value'    => $this->a->get_value_from_post( $name ),
//				'text_on'  => $this->a->__( 'Yes' ),
//				'text_off' => $this->a->__( 'No' ),
//			) ),
//		) );
		
//		$name = 'debug';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'     => $this->a->__( 'Logging verbosity' ),
//			'tooltip' => $this->a->__( 'Set this setting as "Error" to get error messages pertaining to the extension in the OpenCart error log  (recommended setting). "Debug" and "Message" levels add a lot of records in the log file so use them if you think that something wrong with the extension and you want to supply detailed report to the developer' ),
//			'label_for' => 'input-' . $name,
//			'element'   => array(
//				'type'        => 'select',
//				'name'        => $name,
//				'class'       => 'form-control',
//				'active'      => $this->a->get_value_from_post( $name ),
//				'value'       => $option->log_verbosity(),
//				'id'          => 'input-' . $name,
//			),
//			'error'       => isset( $resp[ $name ] ) ?
//			$resp[ $name ] : null,
//		) );
		
		$name = 'test_mode';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Test mode' ),
			'tooltip' => $this->a->__( 'Defines whether the extension is in the test mode. You also need to change corresponding setting of Stripe Dashboard' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
			) ),
		) );
		
//		$name = 'uninstall_clear_settings';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Clear settings' ),
//			'tooltip' => $this->a->__( 'Whether to clear the extension\'s settings during uninstallation' ),
//			'element' => $this->a->r()->render_fancy_checkbox( array(
//				'name'     => $name,
//				'id'       => $name,
//				'value'    => $this->a->get_value_from_post( $name ),
//				'text_on'  => $this->a->__( 'Yes' ),
//				'text_off' => $this->a->__( 'No' ),
//			) ),
//		) );
		
//		$name = 'error_order_notification';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Notify on order error' ),
//			'tooltip' => $this->a->__( 'Whether to send an email notification to the store administrator on payment failure' ),
//			'element' => $this->a->r()->render_fancy_checkbox( array(
//				'name'     => $name,
//				'id'       => $name,
//				'value'    => $this->a->get_value_from_post( $name ),
//				'text_on'  => $this->a->__( 'Yes' ),
//				'text_off' => $this->a->__( 'No' ),
//			) ),
//		) );
		
//		$name = 'three_d_secure';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( '3D Secure' ),
//			'tooltip' => $this->a->__( 'Perform 3D Secure validation' ),
//			'element' => $this->a->r()->render_fancy_checkbox( array(
//				'name'     => $name,
//				'id'       => $name,
//				'value'    => $this->a->get_value_from_post( $name ),
//				'text_on'  => $this->a->__( 'Yes' ),
//				'text_off' => $this->a->__( 'No' ),
//			) ),
//		) );

		$name = 'form_layout';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Payment card fields layout' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'You may select how to display payment card information fields' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => [
					'one_line'   => $this->a->__( 'Display in one line' ),
					'two_line'   => $this->a->__( 'Display in two lines' ),
					'three_line' => $this->a->__( 'Display in three lines' ), 
				],
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );

		$name = 'card_input_border';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Card input border' ) )
			->tooltip( $this->a->__( 'Border width of card input fields' ) )
			->element(
				(new Slider( $this->a->formValue( $name, Slider::def(null,null,1) ) ) )
					->max(10)
					->title( $this->a->__( 'Width in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);

		$name = 'card_input_border_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Card input border color' ),
			'tooltip' => $this->a->__( 'Border color for card input fields' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, 'white' ),
			) ),
		) );

		$name = 'card_input_border_radius';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Card input border radius' ) )
			->tooltip( $this->a->__( 'Border radius of card input fields' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Width in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);

		$name = 'card_input_bg';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Card input background color' ),
			'tooltip' => $this->a->__( 'Background color for card input fields' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, 'white' ),
			) ),
		) );

		$name = 'card_input_text_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Card input text color' ),
			'tooltip' => $this->a->__( 'Text color for card input fields' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, 'black' ),
			) ),
		) );

		$name = 'card_input_padding';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Card input padding' ) )
			->tooltip( $this->a->__( 'Padding of card input fields' ) )
			->element(
				(new Slider( $this->a->formValue( $name, Slider::def(null,null,0) ) ) )
					->max(50)
					->title( $this->a->__( 'Padding in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_width';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Form width' ) )
			->tooltip( $this->a->__( 'Restrict payment form width to some maximum value in pixels' ) )
			->element(
				(new Slider( $this->a->formValue( $name, Slider::def(null,null,500) ) ) )
					->max([ 1000, 100 ])
                    ->text(['px', '%'])
					->title( [$this->a->__( 'Width in px' ), $this->a->__('Width in %%')] )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'border_radius';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Border radius' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Radius in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);

        $name = 'form_header_padding';
        $data[ $name ] = (new FormGroup())
            ->label( $this->a->__( 'Header elements padding' ) )
            ->element(
                (new Slider( $this->a->formValue( $name, Slider::def(null,null,5) ) ) )
                    ->max(50)
                    ->title( $this->a->__( 'Padding in px' ) )
                    ->id( $name )
                    ->name( $name, $this->a )
            );

        $name = 'form_margin_top';
        $data[ $name ] = (new FormGroup())
            ->label( $this->a->__( 'Top margin' ) )
            ->element(
                (new Slider( $this->a->formValue( $name, Slider::def(null,null,0) ) ) )
                    ->max(1000)
                    ->min(-1000)
                    ->title( $this->a->__( 'Margin in px' ) )
                    ->id( $name )
                    ->name( $name, $this->a )
            );

        $name = 'form_header_bg_color';
        $data[$name] = (new FormGroup())
            ->label($this->a->__('Header background color'))
            ->element(
                (new Color())
                    ->name( $name, $this->a )
                    ->id( $name )
                    ->value( $this->a->formValue( $name, '#bbbbbb' ) )
            );

        $name = 'form_header_text_color';
        $data[$name] = (new FormGroup())
            ->label($this->a->__('Header text color'))
            ->element(
                (new Color())
                    ->name( $name, $this->a )
                    ->id( $name )
                    ->value( $this->a->formValue( $name, '#ffffff' ) )
            );

        $name = 'form_text_color';
        $data[$name] = (new FormGroup())
            ->label($this->a->__('Text color'))
            ->element(
                (new Color())
                    ->name( $name, $this->a )
                    ->id( $name )
                    ->value( $this->a->formValue( $name, '#ffffff' ) )
            );
		
		$name = 'form_shadow';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Form\'s shadow' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Shadow in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_shadow_blur';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Form\'s shadow blur' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Blur in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_border';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Border' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(20)
					->title( $this->a->__( 'Border in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_padding';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Form padding' ) )
			->tooltip( $this->a->__( 'Restrict payment form width to some maximum value in pixels' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Padding in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);

		$name = 'confirm_button_width';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Confirm button width' ) )
			->tooltip( $this->a->__( 'You can set size in px or in percents' ) )
			->element(
				(new Slider( $this->a->formValue( $name, Slider::def(null,null,100, 'px') ) ) )
					->max( [1000,100] )
					->title( [$this->a->__( 'Width in px' ), $this->a->__( 'Width in %' )] )
					->text( ['px', '%'] )
					->id( $name )
					->name( $name, $this->a )
			);

		$name = 'confirm_button_alignment';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Confirm button alignment' ) )
			->element(
				(new Select([
					'left'   => $this->a->__( 'Left' ),
					'center' => $this->a->__( 'Center' ),
					'right'  => $this->a->__( 'Right' )
				]) )
					->value( $this->a->get_value_from_post( $name, 'right' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_secondary_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Secondary color' ),
			'tooltip' => $this->a->__( 'Supplementary color which will be applied to buttons, input elements and the like' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, '#204d74' ),
			) ),
		) );
		
		$name = 'form_background';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Form background color' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, '#fff' ),
			) ),
		) );
		
		$name = 'form_font_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Font color' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, '#000' ),
			) ),
		) );
		
		// Animation in
		$name = 'form_animation_in';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Popup animation in' ),
			'element' => $this->a->r( array(
				'type'        => 'select',
				'class'       => 'form-control',
				'id'          => $name,
				'name'        => $name,
				'value'       => [
					'bounceIn'          => $this->a->__( 'BounceIn' ),
					'bounceInDown'      => $this->a->__( 'BounceInDown' ),
					'bounceInLeft'      => $this->a->__( 'BounceInLeft' ),
					'bounceInRight'     => $this->a->__( 'BounceInRight' ),
					'bounceInUp'        => $this->a->__( 'BounceInUp' ),
					'fadeIn'            => $this->a->__( 'FadeIn' ),
					'fadeInDown'        => $this->a->__( 'FadeInDown' ),
					'fadeInDownBig'     => $this->a->__( 'FadeInDownBig' ),
					'fadeInLeft'        => $this->a->__( 'FadeInLeft' ),
					'fadeInLeftBig'     => $this->a->__( 'FadeInLeftBig' ),
					'fadeInRight'       => $this->a->__( 'FadeInright' ),
					'fadeInRightBig'    => $this->a->__( 'FadeInRightBig' ),
					'fadeInUp'          => $this->a->__( 'FadeInUp' ),
					'fadeInUpBig'       => $this->a->__( 'FadeInUpBig' ),
					'rotateIn'          => $this->a->__( 'RotateIn' ),
					'rotateInDownLeft'  => $this->a->__( 'RotateInDownLeft' ),
					'rotateInDownRight' => $this->a->__( 'RotateInDownRight' ),
					'rotateInUpLeft'    => $this->a->__( 'RotateInUpLeft' ),
					'rotateInUpRight'   => $this->a->__( 'RotateInUpRight' ),
					'jackInTheBox'      => $this->a->__( 'JackInTheBox' ),
					'rollIn'            => $this->a->__( 'RollIn' ),
					'zoomIn'            => $this->a->__( 'ZoomIn' ),
					'zoomInDown'        => $this->a->__( 'ZoomInDown' ),
					'zoomInRight'       => $this->a->__( 'ZoomInRight' ),
					'zoomInLeft'        => $this->a->__( 'ZoomInLeft' ),
					'zoomInUp'          => $this->a->__( 'ZoomInUp' ),
					'slideInDown'       => $this->a->__( 'SlideInDown' ),
					'slideInRight'      => $this->a->__( 'SlideInRight' ),
					'slideInLeft'       => $this->a->__( 'SlideInLeft' ),
					'slideInUp'         => $this->a->__( 'SlideInUp' ),
					'lightSpeedIn'      => $this->a->__( 'LightSpeedIn' ),
				],
				'active'      => $this->a->get_value_from_post( $name, 'bounceInDown' ),
			) ),
		) );
		
		// Animation out
		$name = 'form_animation_out';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Popup animation out' ),
			'element' => $this->a->r( array(
				'type'        => 'select',
				'class'       => 'form-control',
				'id'          => $name,
				'name'        => $name,
				'value'       => [
					'bounceOut'          => $this->a->__( 'BounceOut' ),
					'bounceOutDown'      => $this->a->__( 'BounceOutDown' ),
					'bounceOutLeft'      => $this->a->__( 'BounceOutLeft' ),
					'bounceOutRight'     => $this->a->__( 'BounceOutRight' ),
					'bounceOutUp'        => $this->a->__( 'BounceOutUp' ),
					'fadeOut'            => $this->a->__( 'FadeOut' ),
					'fadeOutDown'        => $this->a->__( 'FadeOutDown' ),
					'fadeOutDownBig'     => $this->a->__( 'FadeOutDownBig' ),
					'fadeOutLeft'        => $this->a->__( 'FadeOutLeft' ),
					'fadeOutLeftBig'     => $this->a->__( 'FadeOutLeftBig' ),
					'fadeOutRight'       => $this->a->__( 'FadeOutright' ),
					'fadeOutRightBig'    => $this->a->__( 'FadeOutRightBig' ),
					'fadeOutUp'          => $this->a->__( 'FadeOutUp' ),
					'fadeOutUpBig'       => $this->a->__( 'FadeOutUpBig' ),
					'rotateOut'          => $this->a->__( 'RotateOut' ),
					'rotateOutDownLeft'  => $this->a->__( 'RotateOutDownLeft' ),
					'rotateOutDownRight' => $this->a->__( 'RotateOutDownRight' ),
					'rotateOutUpLeft'    => $this->a->__( 'RotateOutUpLeft' ),
					'rotateOutUpRight'   => $this->a->__( 'RotateOutUpRight' ),
					'hinge'              => $this->a->__( 'Hinge' ),
					'rollOut'            => $this->a->__( 'RollOut' ),
					'zoomOut'            => $this->a->__( 'ZoomOut' ),
					'zoomOutDown'        => $this->a->__( 'ZoomOutDown' ),
					'zoomOutRight'       => $this->a->__( 'ZoomOutRight' ),
					'zoomOutLeft'        => $this->a->__( 'ZoomOutLeft' ),
					'zoomOutUp'          => $this->a->__( 'ZoomOutUp' ),
					'slideOutDown'       => $this->a->__( 'SlideOutDown' ),
					'slideOutRight'      => $this->a->__( 'SlideOutRight' ),
					'slideOutLeft'       => $this->a->__( 'SlideOutLeft' ),
					'slideOutUp'         => $this->a->__( 'SlideOutUp' ),
					'lightSpeedOut'      => $this->a->__( 'LightSpeedOut' ),
				],
				'active'      => $this->a->get_value_from_post( $name, 'bounceOutUp' ),
			) ),
		) );
		
		// Animation out
		$name = 'form_image';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Image' ),
			'element' => $this->a->r( array(
				'type'        => 'select',
				'class'       => 'form-control',
				'id'          => $name,
				'name'        => $name,
				'value'       => [
					Advertikon::FORM_IMAGE_NONE    => $this->a->__( 'Disabled' ),
					Advertikon::FORM_IMAGE_LOGO    => $this->a->__( 'Store Logo' ),
					Advertikon::FORM_IMAGE_PRODUCT => $this->a->__( 'Product Image' ),
					Advertikon::FORM_IMAGE_CUSTOM  => $this->a->__( 'Custom Image' ),
					
				],
				'active'      => $this->a->get_value_from_post( $name, 'bounceOutUp' ),
			) ),
		) );
		
		$name = 'form_image_custom';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Select image' ),
			'element' => $this->a->r( array(
				'type'   => 'image',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name ),
			) ),
		) );
		
		$name = 'form_image_padding';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Image padding' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(200)
					->title( $this->a->__( 'Padding in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_image_border_radius';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Image border radius' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(200)
					->title( $this->a->__( 'Radius in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_image_border_width';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Image border width' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(20)
					->title( $this->a->__( 'Border in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_image_width';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Image width' ) )
			->tooltip( $this->a->__( 'Set to 0 to expand width to 100%' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(1000)
					->title( $this->a->__( 'Width in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_image_top';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Image vertical position' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(1000)
                    ->min(-1000)
					->title( $this->a->__( 'Position in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_image_left';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Image horizontal position' ) )
			->tooltip( $this->a->__( 'Set to 0 to center the image' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
					->max(2000)
					->title( $this->a->__( 'Position in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'form_image_border_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Image border color' ),
			'element' => $this->a->r( array(
				'type'   => 'color',
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, '#fff' ),
			) ),
		) );
		
		$name = 'card_name';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Cardholder\'s name' ),
			'tooltip' => $this->a->__( 'Whether to display a cardholder\'s name field in payment form' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'card_images_show';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Card vendors logos' ),
			'tooltip' => $this->a->__( 'Show logos of supported payment systems' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'preview_image_show';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Show payment card preview' ),
			'tooltip' => $this->a->__( 'Show payment card preview in order to help customer to fill in payment form fields. Only available with two- and three-rows input fields layouts' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'card_images_width';
        $data[ $name ] = (new FormGroup())
            ->label( $this->a->__( 'Vendors images set width' ) )
	        ->tooltip( $this->a->__( 'Width (in %) of available width' ) )
            ->element(
            	(new Slider( $this->a->formValue( $name, [ [], ['value' => 100, 'units' => '%'], ] ) ) )
		            ->max(100)
		            ->title( $this->a->__( 'Width in %' ) )
	                ->id( $name )
		            ->name( $name, $this->a )
            );

		$name = 'card_images_padding';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Vendors images set padding' ) )
			->tooltip( $this->a->__( 'Padding for single item in images set' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 5, 'units' => 'px'], ] ) ) )
					->max(200)
					->title( $this->a->__( 'Padding in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'remember_me';
        $data[ $name ] = (new FormGroup())
            ->label( $this->a->__( 'Save card' ))
            ->tooltip($this->a->__( 'Allows customer to decide whether to save payment card in Stripe Dashboard for further use and select saved card to pay with' ))
            ->element((new FancyCheckBox())
                ->textOn($this->a->__( 'Yes' ))
                ->textOff($this->a->__( 'No' ))
                ->dependsOn('#saved_card_secret')
                ->name($name, $this->a )
                ->id($name)
                ->value($this->a->formValue( $name ))
            );
		
//		$name = 'saved_card_secret';
//		$data[ $name ] = (new FormGroup())
//            ->label($this->a->__( 'Saved card password' ))
//            ->tooltip($this->a->__( 'Customer should supply an additional password for saved card in order to use card in the future' ))
//            ->element((new FancyCheckBox())
//                ->textOn($this->a->__( 'Yes' ))
//                ->textOff($this->a->__( 'No' ))
//                ->dependsOff('#remember_me')
//                ->name($name, $this->a )
//                ->id($name)
//                ->value($this->a->formValue( $name ))
//            );
		
		$name = 'edit_cards';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Customers can edit cards' ),
			'tooltip' => $this->a->__( 'Allow customers to manage saved cards via its accounts' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'edit_subscription';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Customers to cancel subscription' ),
			'tooltip' => $this->a->__( 'Allow customers to cancel subscription' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'cancel_subscription_now';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Cancel subscription now' ),
			'tooltip' => $this->a->__( 'Subscription can be canceled immediately (also by customers). Prorations are not applied' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );

		$name = 'save_customer';
		$data[$name] = (new FormGroup())
            ->label( $this->a->__( 'Save cusotmer' ) )
            ->tooltip( $this->a->__( 'Do save customer to Stripe Dashboard' ) )
            ->element( (new FancyCheckBox() )
                ->dependsOn( '#remember_me' )
                ->dependsOff( '#remember_me' )
                ->value( $this->a->formValue( $name, false ) )
                ->id($name)
                ->name($name, $this->a)
            );
		
		if ( version_compare( VERSION, '3', '<' ) ) {
			$name = 'log_activity';
			$data[ $name ] = $this->a->r()->render_form_group( array(
				'label'   => $this->a->__( 'Log customer activity' ),
				'tooltip' => $this->a->__( 'Record client activity related to the card management into "Customer Activity Report"' ),
				'element' => $this->a->r()->render_fancy_checkbox( array(
					'name'     => $name,
					'id'       => $name,
					'value'    => $this->a->get_value_from_post( $name ),
					'text_on'  => $this->a->__( 'Yes' ),
					'text_off' => $this->a->__( 'No' ),
				) ),
			) );
		}
		
		$name = 'popup';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Popup form' ),
			'tooltip' => $this->a->__( 'Payment form will popup on button click' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
			) ),
		) );
		
//		$name = 'payment_request_button';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Payment Request Button' ),
//			'description' => $this->a->__( 'Enable Apple Pay, Google Pay and Microsoft Pay for supported browsers' ),
//			'element' => $this->a->r()->render_fancy_checkbox( array(
//				'name'     => $name,
//				'id'       => $name,
//				'value'    => \Advertikon\Setting::get( $name, $this->a ),
//				'text_on'  => $this->a->__( 'Yes' ),
//				'text_off' => $this->a->__( 'No' ),
//				'dependent_off' => '#payment_request_button_add',
//			) ),
//		) );
		
//		$name = 'payment_request_button_add';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Payment Request Button with payment form' ),
//			'tooltip' => $this->a->__( 'Show both payment request button and payment form. If disabled - ' .
//					'payment form shows only as a fallback when the browser dosn\'t support Payment Request API' ),
//			'element' => $this->a->r()->render_fancy_checkbox( array(
//				'name'     => $name,
//				'id'       => $name,
//				'value'    => \Advertikon\Setting::get( $name, $this->a ),
//				'text_on'  => $this->a->__( 'Yes' ),
//				'text_off' => $this->a->__( 'No' ),
//				'dependent_on' => '#payment_request_button',
//			) ),
//		) );
		
//		$name = 'payment_request_button_width';
//		$data[ $name ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Payment Request Button width' ),
//			'tooltip' => $this->a->__( 'Width in pixels' ),
//			'element' => [
//				'type' => 'number',
//				'name'   => $name,
//				'id'     => $name,
//				'value'  => \Advertikon\Setting::get( $name, $this->a, [ 'value' => 300 ] ),
//				'class' => 'config-control form-control',
//			],
//		) );

		$name = 'button';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Status' ),
			'tooltip' => $this->a->__( 'Specifies whether to enable the "Pay in one click" button on product page' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    =>$this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );
		
		$name = 'failure_return_page';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Failure page' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Page to redirect a customer to in case of payment failure for payment methods which require payment authorization (Alipay, 3D Secure etc.)' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name, $this->a->u()->catalog_url( true ) ),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'button_name';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Payment method name' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'The description of payment method, to be shown in an order information' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => $this->a->__( 'Payment method name' ),
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'button_shipping';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Shipping method' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Shipping method to apply to all the orders created with "Pay in one click" button. If product requires shipping and this setting is omitted - button won\'t be shown' ),
			'element'   => array(
				'type'        => 'select',
				'name'        => $name,
				'class'       => 'form-control',
				'active'      => $this->a->get_value_from_post( $name ),
				'value'       => $option->shipping_methods(),
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'describe_price';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Describe price' ),
			'tooltip' => $this->a->__( 'Specifies whether to show detailed information pertains price (subtotal, tax, shipping cost) as a list under the button' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );
		
		$name = 'button_text_height';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Text height' ) )
			->tooltip( $this->a->__( 'Height of the button\'s caption' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 20, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Height in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'button_padding_v';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Vertical padding (affects height)' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 10, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Padding in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'button_padding_h';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Horizontal margin' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 10, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Padding in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);

        $name = 'button_margin_vertical';
        $data[ $name ] = (new FormGroup())
            ->label( $this->a->__( 'Vertical margin' ) )
            ->element(
                (new Slider( $this->a->formValue( $name, [ [], ['value' => 10, 'units' => 'px'], ] ) ) )
                    ->max(100)
                    ->title( $this->a->__( 'Margin in px' ) )
                    ->id( $name )
                    ->name( $name, $this->a )
            );

        $name = 'button_margin_horizontal';
        $data[ $name ] = (new FormGroup())
            ->label( $this->a->__( 'Horizontal padding (affects width)' ) )
            ->element(
                (new Slider( $this->a->formValue( $name, [ [], ['value' => 0, 'units' => 'px'], ] ) ) )
                    ->max(100)
                    ->title( $this->a->__( 'Margin in px' ) )
                    ->id( $name )
                    ->name( $name, $this->a )
            );
		
		$name = 'button_radius';
		$data[ $name ] = (new FormGroup())
			->label( $this->a->__( 'Button\'s border radius' ) )
			->element(
				(new Slider( $this->a->formValue( $name, [ [], ['value' => 5, 'units' => 'px'], ] ) ) )
					->max(50)
					->title( $this->a->__( 'Radius in px' ) )
					->id( $name )
					->name( $name, $this->a )
			);
		
		$name = 'button_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Button color' ),
			'tooltip' => $this->a->__( 'A fill color of the payment button' ),
			'element' => $this->a->r()->render_color( array(
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, '#2197c4' ),
				'class'  => 'iris',
			) ),
		) );
		
		$name = 'button_text_color';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Button\'s text color' ),
			'tooltip' => $this->a->__( 'Color of the payment button\'s caption' ),
			'element' => $this->a->r()->render_color( array(
				'name'   => $name,
				'id'     => $name,
				'value'  => $this->a->get_value_from_post( $name, '#ffffff' ),
				'class'  => 'iris',
			) ),
		) );
		
		$name = 'button_full_width';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'   => $this->a->__( 'Full width' ),
			'tooltip' => $this->a->__( 'Layout of the button becomes responsive - filling in all available width' ),
			'element' => $this->a->r()->render_fancy_checkbox( array(
				'name'     => $name,
				'id'       => $name,
				'value'    => $this->a->get_value_from_post( $name ),
				'text_on'  => $this->a->__( 'Yes' ),
				'text_off' => $this->a->__( 'No' ),
			) ),
		) );
		
		$name = 'webhook_url';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Webhook\'s URL' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'URL to use as a webhook endpoint for your site in Stripe Dashboard' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control clipboard',
				'custom_data' => 'readonly="readonly"',
				'value'       => $this->a->u()->catalog_url( 'auto' ) . 'index.php?route=' . $this->a->full_name . '/webhooks',
				'id'          => 'input-' . $name,
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );

		// $name = 'webhook_secret';
		// $data[ $name ] = $this->a->r()->render_form_group( array(
		// 	'label'     => $this->a->__( 'Webhook\'s signing secret' ),
		// 	'label_for' => 'input-' . $name,
		// 	'tooltip'   => $this->a->__( 'Secret to sign messages to specific webhook endpoint. Can be found at Stripe Dashboard Webhooks section' ),
		// 	'element'   => array(
		// 		'type'        => 'text',
		// 		'name'        => $name,
		// 		'class'       => 'form-control',
		// 		'value'       => $this->a->get_value_from_post( $name ),
		// 		'id'          => 'input-' . $name,
		// 	),
		// 	'error'       => isset( $resp[ $name ] ) ?
		// 	$resp[ $name ] : null,
		// ) );
		
		$name = 'create_subscription_callback';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Subscription creation callback' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'URL to send callback to on a subscription creation. (Optional)' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => 'http://callback_url.com',
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'create_subscription_callback_data';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Creation callback data' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Additional data in a form of foo=bar, baz=boo, to be sent as POST query along with: oc_customer, stripe_customer, oc_subscription, stripe_subscription, status (new) to "Subscription creation callback" URL. (Optional)' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => 'foo=bar,baz=boo',
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'cancel_subscription_callback';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Subscription cancellation callback' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'URL to send callback to on subscription cancellation. (Optional)' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => 'http://callback_url.com',
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$name = 'cancel_subscription_callback_data';
		$data[ $name ] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Cancellation callback data' ),
			'label_for' => 'input-' . $name,
			'tooltip'   => $this->a->__( 'Additional data in a form of foo=bar, baz=boo, to be sent as POST query along with: oc_customer, stripe_customer, oc_subscription, stripe_subscription, status (cancel) to "Subscription cancellation callback" URL. (Optional)' ),
			'element'   => array(
				'type'        => 'text',
				'name'        => $name,
				'class'       => 'form-control',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => 'input-' . $name,
				'placeholder' => 'foo=bar,baz=boo',
			),
			'error'       => isset( $resp[ $name ] ) ?
			$resp[ $name ] : null,
		) );
		
		$id = 'template-adk_stripe_template_error_order';
		$name = $this->a->build_name( $id, '-' );
		$data['template'] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Payment error template' ),
			'label_for' => $name,
			'tooltip'   => $this->a->__( 'The email message template to send to the store owner on payment failure. List of supported shortcodes: %s', implode( ', ', $option->shortcode() ) . '. Add {error} shortcode to display error message' ),
			'element'   => array(
				'type'        => 'textarea',
				'name'        => $name,
				'class'       => 'form-control summernote',
				'value'       => $this->a->get_value_from_post( $name ),
				'id'          => $name,
			),
			'error'       => isset( $resp['input_errors']['template']['adk_stripe_template_errro_order'] ) ?
			$resp['input_errors']['template']['adk_stripe_template_errro_order'] : null,
		) );
		
		$data['select_stripe_account'] = $this->a->r()->render_form_group( array(
			'label'     => $this->a->__( 'Select Stripe account' ),
			'label_for' => 'select-account',
			'element'   => array(
				'type'        => 'select',
				'class'       => 'form-control',
				'active'      => null,
				'value'       => StripeAccount::allNames(),
				'id'          => 'select-account',
			),
			'class'     => 'static',
		) );

//		$locale = [];
//		$this->a->trigger( 'locale', $locale );
		
//		$locale = array_merge( $locale, array(
//			'variables'                 => '{' . implode( '},{' , array_keys( $shortcode->get_shortcode_data() ) ) . '}',
//			'accountDetailsTemplate'    => $this->get_stripe_account_field(),
//			'errorNameExists'           => $this->a->__( 'Name already exists' ),
//			'errorAcountSameCurrency'   => $this->a->__( 'Two accounts can not have the same currency' ),
//			'imgStripeUrl'              => $this->a->image_url,
//			'dirLogs'                   => DIR_LOGS,
//			'currency'                  => array_values( $option->currency() ),
//			'sureRemoveLost'            => $this->a->__( 'Remove record ?' ),
//			'isAppleVerified'           => !class_exists( 'Advertikon\Stripe\Extended' ) ||
//												\Advertikon\Setting::get( 'is_apple_verified', $this->a ),
//			'appleVerifyUrl'            => $this->a->u( 'verify_apple' ),
//			'modalHeader'               => 'Stripe',
//			'formCustomImageMode'       => \Advertikon\Stripe\Advertikon::FORM_IMAGE_CUSTOM,
//			'registerWebhookUrl'        => $this->a->u( 'register_webhook' ),
//			//'isWebhook'                 => (bool)\Advertikon\Setting::get( 'webhook_secret', $this->a ),
//		) );
//
//		$data['locale'] = json_encode( $locale );
		
		$data['accounts'] = array();
		$data['a']        = $this->a;

		foreach( $this->a->get_value_from_post( 'account', array( 'default' => array() ) ) as $name => $account ) {
			$data['accounts'][] = (new \Advertikon\Stripe\StripeAccount($name, $account))->fields( $resp );
		}
		
		$this->get_label_controls( $data );

		return $data;
	}

    /**
     * @param $data
     * @throws \Advertikon\Exception
     */
	protected function get_label_controls( &$data ) {
		$translator = new \Advertikon\Translator( $this->a );

		$name = 'caption_script_error';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Script error' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The error message to show in case of an unrecoverable script error (payment cannot be fulfilled)' )
		] );

		$name = 'caption_charge_value';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Different currency' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The notification to show if the charge currency differs from the storefront currency (eg when you set \'Payment currency\' setting to \'Store\' and default store currency differs from the storefront currency). {{amount}} placeholder may be used to show real amount' )
		] );
		
		$name = 'caption_form_cardholder_name';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Cardholder\'s name' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The label to the cardholder\'s name input of the payment form' )
		] );
		
		$name = 'caption_form_rerember_me';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Save card' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The label to "Save card" checkbox of the payment form' )
		] );
		
		$name = 'caption_form_rerember_me_description';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Save card description' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Description text to "Save card" checkbox of the payment form' )
		] );
		
		$name = 'caption_form_make_default';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Make default' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Label to "Make saved card default" checkbox of the payment form' )
		] );
		
//		$name = 'caption_form_make_default_description';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Make default description' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'Description text to "Make saved card default" checkbox of the payment form' )
//		] );
		
		$name = 'caption_form_select_card';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Choose saved card' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Label to the drop-down list of saved cards' )
		] );
		
//		$name = 'caption_form_card_password';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Saved card password' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'Label to the saved card\'s password input' )
//		] );
		
//		$name = 'caption_form_card_password_description';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Password description' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'Description text to the saved card\'s password input' )
//		] );
//
//		$name = 'caption_wait_script';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Page loading' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'The notification asking a customer to wait till the page fully loads if he or she tries to make a payment (in case of slow Internet connection)' )
//		] );
		
		$name = 'caption_payment_error';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Payment error' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Error message on the payment gateway failure' )
		] );
		
		$name = 'caption_payment_success';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Successful payment' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Successful payment notification' )
		] );
		
		$name = 'caption_order_placing';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Order placing' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The message which displays during an order processing' )
		] );
		
//		$name = 'caption_token_create';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Token creation' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'The message which displays during the card details tokenization' )
//		] );
		
		$name = 'caption_button_placing';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Pay button caption' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The caption of "Pay-in-one-click" button which shows during payment processing' )
		] );
		
//		$name = 'caption_unknown_vendor';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Unknown vendor' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'The error message which displays on an attempt to pay with the card which is not supported by Stripe' )
//		] );
		
		$name = 'caption_forbidden_vendor';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Forbidden vendor' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The error message to show on an attempt to pay with a card forbidden by extension\'s configuration' )
		] );
		
//		$name = 'caption_error_card_password_save';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Saved card password' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'The error message to show when the additional password is mandatory in order to save payment card' )
//		] );
		
//		$name = 'caption_error_card_password_use';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Saved card password (use)' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'The error message which says that customer needs to specify the password in order to use saved payment card' )
//		] );

//		$name = 'caption_payment_request_button';
//		$data[ $name ] = $this->a->r()->render_form_group( [
//			'label'   => $this->a->__( 'Payment Request Button label' ),
//			'element' => $translator->render_translate_control( $name ),
//			'tooltip' => $this->a->__( 'Text to place between payment request button (Apple Pay, ' .
//					'Google Pay, Microsoft Pay) and payment form in case they both are displayed' )
//		] );

		$name = 'caption_button_text';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'One click button\'s caption' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The label on the "Pay in one click" button. If you include {{amount}} to the label text, it will be replaced with the charge amount (in corresponding currency)' )
		] );

		$name = 'caption_button_caption';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Payment button\'s caption' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'The label on the payment button of the pop-up form. If you include {{amount}} to the label text, it will be replaced with a charge amount (in corresponding currency)' )
		] );

        $name = 'caption_confirm_caption';
        $data[ $name ] = $this->a->r()->render_form_group( [
            'label'   => $this->a->__( 'Confirm button caption' ),
            'element' => $translator->render_translate_control( $name ),
            'tooltip' => $this->a->__( 'The label on the Confirm button' )
        ] );

		$name = 'caption_form_caption';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Form\'s caption' ),
			'element' => $translator->render_translate_control( $name ),
		] );

		$name = 'caption_title';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Title' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Checkout title' )
		] );
		
		$name = 'caption_sandbox_title';
		$data[ $name ] = $this->a->r()->render_form_group( [
			'label'   => $this->a->__( 'Sandbox Title' ),
			'element' => $translator->render_translate_control( $name ),
			'tooltip' => $this->a->__( 'Checkout title when the extension is in the test mode' )
		] );
	}
	
//	/**
//	 * Get stripe account template for administrative area
//	 * @param array $data Template data
//	 * @param string $teomlate_name Template name
//	 * @return String
//	 */
//	public function get_stripe_account_field( $data = array(), $template_name = null ) {
//		$templ = false;
//
//		if ( is_null( $template_name ) ) {
//			$template_name = '{template_name}';
//			$templ = true;
//		}
//
//		extract( $data );
//
//		$data['default'] = $template_name === 'default';
//		$option = new \Advertikon\Stripe\Option();
//		$name_prefix = "account-" . $template_name;
//
//		//Account name field
//		$field = 'account_name';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data['account_name'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Account name' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control account_name',
//				'placeholder' => $this->a->__( 'Account name' ),
//				'value'       => $this->a->get_value_from_post( $name, 'New' ),
//				'custom_data' => [ 'data-name' => \Advertikon\Setting::prefix_name( $name, $this->a ), ],
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['account_name'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['account_name'] : null,
//			'css' => !class_exists( 'Advertikon\Stripe\Extended' ) ? 'display: none' : '',
//		) );
//
//		// Account currency field
//		$field = 'account_currency';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data['account_currency'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Account currency' ),
//			'element' => $this->a->r( array(
//				'type'        => 'select',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control account-currency',
//				'active'      => $this->a->get_value_from_post( $name ),
//				'value'       => $option->currency_code(),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'tooltip' => $this->a->__( 'Front end currency, associated with this Stripe account' ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['account_currency'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['account_currency'] : null,
//			'css' => !class_exists( 'Advertikon\Stripe\Extended' ) ? 'display: none' : '',
//		) );
//
//		// Test secret key field
//		$field = 'test_secret_key';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data['test_secret_key'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Test secret key' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control secret-test secret-secret',
//				'placeholder' => $this->a->__( 'Test secret key' ),
//				'value'       => $this->a->get_value_from_post( $name ),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['test_secret_key'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['test_secret_key'] : null,
//		) );
//
//		// Test public key field
//		$field = 'test_public_key';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data['test_public_key'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Test publishable key' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control secret-test secret-public',
//				'placeholder' => $this->a->__( 'Test publishable key' ),
//				'value'       => $this->a->get_value_from_post( $name ),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['test_public_key'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['test_public_key'] : null,
//		) );
//
//		// Live secret key field
//		$field = 'live_secret_key';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id , '-');
//
//		$data['live_secret_key'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Live secret key' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control secret-live secret-secret',
//				'placeholder' => $this->a->__( 'Live secret key' ),
//				'value'       => $this->a->get_value_from_post( $name ),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['live_secret_key'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['live_secret_key'] : null,
//		) );
//
//		// Live public key field
//		$field = 'live_public_key';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data['live_public_key'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Live publishable key' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control secret-live secret-public',
//				'placeholder' => $this->a->__( 'Live publishable key' ),
//				'value'       => $this->a->get_value_from_post( $name ),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['live_public_key'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['live_public_key'] : null,
//		) );
//
//		// Live signing key
//		$field = 'live_signing_key';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data['live_signing_key'] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Live signing secret' ),
//			'tooltip'   => $this->a->__( 'Webhook signing secret for live mode. Can be found at Developers > Webhooks section of Stripe Dashboard when viewing live data.' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control secret-signing secret-live',
//				'placeholder' => $this->a->__( 'Live signing secret' ),
//				'value'       => $this->a->get_value_from_post( $name ),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ]['live_signing_key'] ) ?
//			$data['error_input_errors']['account'][ $template_name ]['live_signing_key'] : null,
//		) );
//
//		// Public signing key
//		$field = 'test_signing_key';
//		$id = $name_prefix . '-' . $field;
//		$name = $this->a->build_name( $id, '-' );
//
//		$data[ $field ] = $this->a->r()->render_form_group( array(
//			'label'   => $this->a->__( 'Test signing secret' ),
//			'tooltip'   => $this->a->__( 'Webhook signing secret for test mode. Can be found at Developers > Webhooks section of Stripe Dashboard when viewing test data' ),
//			'element' => $this->a->r( array(
//				'type'        => 'text',
//				'id'          => $id,
//				'name'        => $templ ? '' : $name,
//				'class'       => 'form-control secret-signing secret-test',
//				'placeholder' => $this->a->__( 'Test signing secret' ),
//				'value'       => $this->a->get_value_from_post( $name ),
//				'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
//			) ),
//			'error'   => isset( $data['error_input_errors']['account'][ $template_name ][ $field ] ) ?
//			$data['error_input_errors']['account'][ $template_name ][ $field ] : null,
//		) );
//
//		$data['a'] = $this->a;
//
//		return $this->a->load->view(
//			$this->a->get_template( $this->a->type . '/advertikon/stripe/account' ), $data
//		);
//	}

    /**
     * @param $secret
     * @return mixed|null
     * @throws \Advertikon\Exception
     * @throws \Stripe\Error\Api
     * @throws Exception
     */
	public function register_webhook( $secret ) {
		$url_1 = $this->a->u()->catalog_url( 'auto' ) . 'index.php?route=' . $this->a->full_name . '/webhooks';
		$url_2 = $this->a->u()->catalog_url( true ) . 'index.php?route=' . $this->a->full_name . '/webhooks';
		$url_3 = $this->a->u()->catalog_url( false ) . 'index.php?route=' . $this->a->full_name . '/webhooks';
		$webhook = null;

		Stripe::setApiKey( $secret );

		foreach( \Stripe\WebhookEndpoint::all()->data as $w ) {
			if ( in_array( $w['url'], [ $url_1, $url_2, $url_3 ] ) ) {
				$webhook = $w;
				break;
			}
		}

		if ( !$webhook ) {
			$webhook = \Stripe\WebhookEndpoint::create( [
				'url' => $url_1,
				'enabled_events' => Advertikon::EVENTS
			] );

			return $webhook->secret;

		} else if ( 'enabled' !== $webhook->status ) {
			throw new \Exception( $this->a->__(
				'Webhook endpoint (secret key %s) is disabled. You need to enable it from Stripe Dashboard',
				$this->a->obscure_str( $secret )
			) );
		}

		return null;
	}
}
