<?php
/**
 * Catalog Advertikon Stripe Controller
 * @author Advertikon
 * @package Stripe
  * @version 5.0.44    
 * 
 * @source catalog/view/theme/default/stylesheet/advertikon/stripe/*
 * #source catalog/view/theme/default/stylesheet/advertikon/advertikon.css
 * 
 * #source catalog/view/javascript/advertikon/advertikon.js
 * @source catalog/view/javascript/advertikon/stripe/*
 * @source catalog/view/javascript/advertikon/qrcode.min.js
 * @source catalog/view/theme/default/stylesheet/advertikon/fa/*
 * 
 * @source catalog/view/theme/default/template/extension/payment/advertikon/stripe/*
 */

use Advertikon\Element\Div;
use Advertikon\Element\Text;
use Advertikon\Setting;
use Advertikon\Store;
use Advertikon\Stripe\Advertikon;
use Advertikon\Stripe\Button;
use Advertikon\Stripe\Extended;
use Advertikon\Stripe\Order;
use Advertikon\Stripe\OrderPrice;
use Advertikon\Stripe\PaymentForm;
use Advertikon\Stripe\Recurring;
use Advertikon\Stripe\Step;
use Advertikon\Stripe\Customer;

/**
 * Class ControllerExtensionPaymentAdvertikonStripe
 * @property DB $db
 * @property Request $request
 * @property Response $response
 * @property Config $config
 * @property Document $document
 * @property Loader $load
 * @property Language $language
 * @property Session $session
 * @property Url $url
 * @property \Cart\Customer $customer
 * @property \Cart\Cart $cart
 * @property \Cart\Currency $currency
 * @property Log $log
 *
 * @property ModelCheckoutOrder $model_checkout_order
 * @property ModelAccountActivity $model_account_activity
 */
class ControllerExtensionPaymentAdvertikonStripe extends Controller {
	use \Advertikon\Trait_Controller;

	/** @var \Advertikon\Stripe\Advertikon */
	public $a = null;

	/** @var ModelExtensionPaymentAdvertikonStripe */
	public $model = null;

	/**
	 * ControllerExtensionPaymentAdvertikonStripe constructor.
	 * @param $registry
	 * @throws \Advertikon\Exception
	 */
	public function __construct( $registry ) {
		parent::__construct( $registry );
		$this->a = Advertikon::instance( $registry );
		$this->model = $this->a->load->model( $this->a->full_name );
	}

	/**
	 * Embed payment form
	 * @return string
	 * @throws \Advertikon\Exception
	 * @throws Exception
	 */
	public function index() {
		$errorWrapper = (new Div())
			->style()->color('#ffffff')->backgroundColor('#F11F1F')->fontWight('bold')
			->textAlign('center')->padding('5px')->stop()->stop();

		try {
			$this->a->setApiKey();
			$orderPrice = new OrderPrice();
			$paymentForm = new PaymentForm();

			$currency       = $orderPrice->currency();
			$total          = $orderPrice->total() + $orderPrice->recurring();
			$totalFormatted = $this->currency->format( $total, $currency, 1, true );

			$data['button_confirm'] = $this->language->get('button_confirm');
			$data['button_back']    = $this->language->get('button_back');

			$data['requireJs'] = $this->a->requireJs( ['stripe/main' => 'main' ] );
			$data['url'] = $this->a->url();

			$data['is_extended'] = $this->a->isExtended;
			$data['total']           = $total;
			$data['currency']        = $currency;
			$data['total_formatted'] = $totalFormatted;
			$data['is_recurring'] = $this->cart->hasRecurringProducts();
			$data['popup'] = $paymentForm->isPopUp();

			$data = array_merge( $data, $paymentForm->getData());

			// String displaying charge real currency
			$data['real_currency'] = $orderPrice->isDifferentCurrency() && $this->a->__( 'caption_charge_value' ) ?
				str_replace( '{{amount}}', $totalFormatted, $this->a->__( 'caption_charge_value' ) ) : '';

			$template = $paymentForm->isPopUp() ? '/advertikon/stripe/pop_up_form' : '/advertikon/stripe/embedded_form';

			return $this->a->load->view( $this->a->get_template( $this->a->type . $template ) , $data );

		} catch( \Advertikon\Stripe\Exception $e ) {
			$this->a->error( $e );
			$errorWrapper->children( new Text( $e->getMessage() ) );
			return $errorWrapper->__toString();

		} catch ( Exception $e ) { 
			$this->a->error( $e );
			$errorWrapper->children( new Text( $this->a->__( 'caption_script_error' ) ) );
			return $errorWrapper->__toString();
		}
	}

    /**
     * @throws \Advertikon\Exception
     */
	public function step() {
		$ret = [];
        $this->a->setApiKey();

		try {
			$step = new Step();
			$ret['step'] = $step->next();

		} catch ( \Advertikon\Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Stripe\Error\Base $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $this->a->__( 'caption_script_error' );
		}

		$this->a->json_response( $ret );
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	public function set_step() {
		$ret = [];
		$this->a->setApiKey();

		$cartId    = isset( $this->request->request['cart_id'] ) ? $this->request->request['cart_id'] : null;
		$invoiceId = isset( $this->request->request['invoice_id'] ) ? $this->request->request['invoice_id'] : null;
		$oneTime   = isset( $this->request->request['one_time'] ) ? $this->request->request['one_time'] : null;

		try {
			$step = new Step();

			if ( is_null( $cartId ) && is_null( $oneTime ) ) {
				throw new Exception( "Step data missing" );
			}

			if ( !is_null( $cartId ) ) {
				if ( !is_null( $invoiceId ) ) {
					$step->failedCartId( $cartId );
					$step->failedInvoice( $invoiceId );

				} else {
					$step->processedSubscription( $cartId );
				}
			}

			if ( !is_null( $oneTime ) ) {
				if ( $oneTime === '1' ) {
					$step->oneTimePaid();

				} else {
					$step->oneTimePaid( false );
				}
			}

			$ret['success'] = 'ok';

		} catch ( \Advertikon\Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Stripe\Error\Base $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $this->a->__( 'caption_script_error' );
		}

		$this->a->json_response( $ret );
	}

	public function create_intent() {
	    try {
	    	$ret = [];
	        $this->a->setApiKey();
	        $cardMethod = new \Advertikon\Stripe\PaymentOption\Card();

	        $saveCard   = !empty( $this->request->request['saveCustomer'] );
	        $savedCard  = isset( $this->request->request['savedCard'] ) ? $this->request->request['savedCard'] : null;
	        $step = new Step();

	        $data = ['metadata' => ['integration_check' => 'accept_a_payment'] ];

	        if ( $savedCard ) {
				$cardMethod->savedCardCase( $savedCard, $data );

	        } else if ( $saveCard || Setting::get( 'save_customer', $this->a, false ) ) {
	        	$cardMethod->saveCardCase($data );
	        }

	        $intentData = $cardMethod->getIntentData( $data );

	        $this->a->log( 'Intent data', $intentData );
            $intent = \Stripe\PaymentIntent::create( $intentData );
		    $this->a->log( $intent );

            $ret['secret'] = $intent->client_secret;
            $step->oneTimePaid();

        } catch ( \Advertikon\Exception $e ) {
	    	$this->a->error( $e );
	        $ret['error'] = $e->getMessage();

        } catch ( Stripe\Error\Base $e ) {
		    $this->a->error( $e );
	        $ret['error'] = $e->getMessage();

        } catch ( Exception $e ) {
	        $this->a->error( $e );
	        $ret['error'] = $this->a->__( 'caption_script_error' );
        }

        $this->a->json_response( $ret );
    }

    public function create_source_intent() {
        try {
            $ret = [];
            $this->a->setApiKey();
            $methodTag = $this->a->requestParam( 'tag' );
            $method = \Advertikon\Stripe\PaymentOption::get( $methodTag );

            if ( is_null( $method ) ) {
                throw new Exception( 'Unknown payment option' );
            }

            $intentData = $method->getIntentData();

            $this->a->log( 'Intent data', $intentData );
            $intent = \Stripe\PaymentIntent::create( $intentData );
            $this->a->log( $intent );

            $ret['secret'] = $intent->client_secret;

        } catch ( \Advertikon\Exception $e ) {
            $this->a->error( $e );
            $ret['error'] = $e->getMessage();

        } catch ( Stripe\Error\Base $e ) {
            $this->a->error( $e );
            $ret['error'] = $e->getMessage();

        } catch ( Exception $e ) {
            $this->a->error( $e );
            $ret['error'] = $this->a->__( 'caption_script_error' );
        }

        $this->a->json_response( $ret );
    }

    public function create_request_intent() {
        try {
            $ret = [];
            $this->a->setApiKey();
            $method = new \Advertikon\Stripe\PaymentOption\PaymentRequest();

            $intentData = $method->getIntentData();

            $this->a->log( 'Intent data', $intentData );
            $intent = \Stripe\PaymentIntent::create( $intentData );
            $this->a->log( $intent );

            $ret['secret'] = $intent->client_secret;

        } catch ( \Advertikon\Exception $e ) {
            $this->a->error( $e );
            $ret['error'] = $e->getMessage();

        } catch ( Stripe\Error\Base $e ) {
            $this->a->error( $e );
            $ret['error'] = $e->getMessage();

        } catch ( Exception $e ) {
            $this->a->error( $e );
            $ret['error'] = $this->a->__( 'caption_script_error' );
        }

        $this->a->json_response( $ret );
    }

	public function create_customer() {
		try {
			$this->a->setApiKey();
			$customer = \Advertikon\Stripe\Customer::get();
			$ret = [];

			if ( $customer->exists() ) {
				$ret['customer'] = $customer->stripeId();

			} else {
				$data = [];
				(new \Advertikon\Stripe\PaymentOption\Card())->saveCardCase( $data );
				$ret['customer'] = $data['customer'];
			}

		} catch ( \Advertikon\Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Stripe\Error\Base $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $this->a->__( 'caption_script_error' );
		}

		$this->a->json_response( $ret );
	}

	public function create_subscription() {
	    $ret = [];

		try {
			$this->a->setApiKey();
			$paymentMethod = isset( $this->request->request['paymentMethod'] ) ?
                $this->request->request['paymentMethod'] : null;

			$customer = isset( $this->request->request['customer'] ) ? $this->request->request['customer'] : null;
			$subscription = null;

			\Advertikon\Stripe\Customer::bindPaymentMethod( $customer, $paymentMethod );
			$step = new Step();

			foreach( $this->cart->getRecurringProducts() as $product ) {
			    if ( !$step->isProcessedSubscription( $product['cart_id'] ?? $product['key'] ) ) {
			        $recurring = new Recurring();
			        $subscription = $recurring->create( $product, $customer );
			        $this->a->log( $subscription );

			        if ( in_array( $subscription->status, [ 'active', 'trialing' ] ) ) {
                        $step->processedSubscription( $product['cart_id'] ?? $product['key'] );
                        $ret['status'] = "succeeded";

                    } else  {
                        if ( $subscription->latest_invoice->payment_intent->status === 'requires_payment_method' ) {
                            $step->failedInvoice($subscription->latest_invoice->id);
                            $step->failedCartId( $product['cart_id'] ?? $product['key'] );
                            $ret['error_message'] = $subscription->latest_invoice->payment_intent->last_payment_error->message;
                        }

                        $ret['status'] = $subscription->latest_invoice->payment_intent->status;
                        $ret['secret'] = $subscription->latest_invoice->payment_intent->client_secret;
                        $ret['invoice_id'] = $subscription->latest_invoice->id;
                    }

			        $ret['subscription_id'] = $subscription->id;
			        $ret['cart_id'] = $product['cart_id'] ?? $product['key'];

			        break;
                }
            }

			if ( is_null( $subscription ) ) {
			    throw new Exception( 'No matching products found' );
            }

		} catch ( \Advertikon\Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Stripe\Error\Base $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $this->a->__( 'caption_script_error' );
		}

		$this->a->json_response( $ret );
	}

	public function retry_invoice() {
		$ret = [];

		try {
			$this->a->setApiKey();
			$paymentMethod = isset( $this->request->request['paymentMethod'] ) ?
                $this->request->request['paymentMethod'] : null;

			$customer = isset( $this->request->request['customer'] ) ? $this->request->request['customer'] : null;

			\Advertikon\Stripe\Customer::bindPaymentMethod( $customer, $paymentMethod );
			$step = new Step();

			$invoice = \Stripe\Invoice::retrieve( [
				'id'     => $step->getFailedInvoice(),
				'expand' => ['payment_intent'],
			] );

			$invoice->pay();

			$step->failedInvoice( null );
            $step->processedSubscription( $step->getFailedCartId() );
            $step->failedCartId(null);
			$this->a->log( $invoice );

		} catch ( \Advertikon\Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Stripe\Error\Base $e ) {
			$this->a->error( $e );
			$ret['error'] = $e->getMessage();

		} catch ( Exception $e ) {
			$this->a->error( $e );
			$ret['error'] = $this->a->__( 'caption_script_error' );
		}

		$this->a->json_response( $ret );
	}

	/**
	 * Saved cards management page
	 * @throws \Advertikon\Exception
	 */
	public function account_cards() {
		if ( !$this->a->isExtended ) {
			return;
		}

		if ( !$this->customer->isLogged() ) {
			$this->session->data['redirect'] = $this->a->u()->url( 'account_cards' );
			$this->response->redirect( $this->a->u()->url( 'account/login' ) );
		}

		$this->document->setTitle( $this->a->__( 'My cards' ) );
		$this->document->addStyle( $this->a->u()->catalog_css( 'advertikon/advertikon.css' ) );
		$this->document->addStyle( $this->a->u()->catalog_css( 'advertikon/animate.min.css' ) );

		$data['breadcrumbs'] = array();

		$data['breadcrumbs'][] = array(
			'text' => $this->a->__( 'Home' ),
			'href' => $this->url->link( 'common/home' ),
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->a->__( 'Account' ),
			'href' => $this->url->link( 'account/account', '', 'SSL' )
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->a->__( 'My cards' ),
			'href' => $this->url->link( 'payment/', '', 'SSL')
		);

		$data['heading_title'] = $this->a->__( 'My cards' );
		$data['button_back'] = $this->a->__( 'Back' );

		$data['back'] = $this->a->u( 'account/account' );
		$data['cards'] = [];
		$data['default_card'] = null;

		if( $this->customer->isLogged() ) {
			$aCustomer = Customer::get();

			if ( $aCustomer->exists() ) {
				$data['default_card'] = $aCustomer->defaultPaymentMethod();

				foreach( $aCustomer->cards() as $card ) {
					$data['cards'][] = Extended::getCardInfo( $card );
				}
			}
		}

		$data['column_left']    = $this->load->controller( 'common/column_left' );
		$data['column_right']   = $this->load->controller( 'common/column_right' );
		$data['content_top']    = $this->load->controller( 'common/content_top' );
		$data['content_bottom'] = $this->load->controller( 'common/content_bottom' );
		$data['footer']         = $this->load->controller( 'common/footer' );
		$data['header']         = $this->load->controller( 'common/header' );
		$data['deleteUrl']      = $this->a->u()->url( 'delete_card' );
		$data['defaultUrl']     = $this->a->u()->url( 'default_card' );

		$locale = [
			'deleteCard' => $this->a->__( 'Do you really want to delete the card?' ),
		];

		$data['requireJs'] = $this->a->requireJs( ['stripe/account_cards' => 'account_cards' ], [], $locale );

		$this->response->setOutput(
			$this->a->load->view(
				$this->a->get_template( $this->a->type . '/advertikon/stripe/account_cards' ), $data
			)
		);
	}

	/**
	 * Delete saved card action
	 * @return void
	 * @throws Exception
	 */
	public function delete_card() {
		$ret = array();
		$card_name = '';

		$log = $this->a->config( 'log_activity' ) && version_compare( VERSION, '3', '<' );

		if( $log ) {
			$this->load->model( 'account/activity' );
		}

		try {
			if( !$this->customer->isLogged() ) {
				$log = false;
				throw new \Advertikon\Exception( $this->a->__( 'Current session has expired' ) );
			}

			if( empty( $this->request->request['card_id'] ) ) {
				throw new \Advertikon\Exception( $this->a->__( 'Card ID is missing' ) );
			}

			$card_id = $this->request->request['card_id'];
			$aCustomer = \Advertikon\Stripe\Customer::get();

			if ( !$aCustomer->exists() ) {
				throw new \Advertikon\Exception( $this->a->__( 'You have no saved cards' ) );
		    }

			$method = $aCustomer->card( $card_id );
			$card_name = $method->card->brand . ' **** ' . $method->card->last4 . ' (' . $method->card->exp_month . '/' . $method->card->exp_year . ')';
			$aCustomer->deleteCard( $card_id );

			if( $log ) {
	    		$activity_data = array(
					'customer_id' => $this->customer->getId(),
					'name'        => $this->customer->getFirstName() . ' ' . $this->customer->getLastName(),
					'card'        => $card_name,
				);

				$this->model_account_activity->addActivity( 'delete_card', $activity_data );
			}

    		$ret['success'] = $this->a->__( 'The card has been deleted' );

		} catch ( \Exception $e ) {
			// Log activity
			if( $log ) {
				$activity_data = array(
					'customer_id' => $this->customer->getId(),
					'name'        => $this->customer->getFirstName() . ' ' . $this->customer->getLastName(),
					'card_name'   => $card_name,
					'message'     => $e->getMessage(),
				);

				$this->model_account_activity->addActivity( 'fail_delete_card', $activity_data );
			}

			$ret['error'] = $e->getMessage();
			$this->a->error( $e );
		}

		$this->response->setOutput( json_encode( $ret ) );
	}

	/**
	 * Make saved card default
	 * @return void
	 * @throws Exception
	 */
	public function default_card() {
		$ret = [];
		$log = $this->a->config( 'log_activity' );

		if( $log ) {
			$this->load->model( 'account/activity' );
		}

		try {
			if( !$this->customer->isLogged() ) {
				$log = false;
				throw new \Advertikon\Exception( $this->a->__( 'Current session has expired' ) );
			}

			if( empty( $this->request->request['card_id'] ) ) {
				throw new \Advertikon\Exception( $this->a->__( 'Card ID is missing' ) );
			}

			$card_id = $this->request->request['card_id'];
			$aCustomer = Customer::get();

			if ( !$aCustomer->exists() ) {
				throw new \Advertikon\Exception( $this->a->__( 'You have no saved cards' ) );
		    }

			$currentDefaultMethod = $aCustomer->defaultPaymentMethod();
			$oldCard = 'None';

			if ( $currentDefaultMethod ) {
				$card = $aCustomer->card( $currentDefaultMethod );
				$oldCard = $card->card->brand . ' **** ' . $card->card->last4 .
					' (' . $card->card->exp_month . '/' . $card->card->exp_year . ')';
			}

			$card = $aCustomer->card( $card_id );
			$newCard = $card->brand . ' **** ' . $card->last4 . ' (' . $card->exp_month . '/' . $card->exp_year . ')';
    		$aCustomer->defaultPaymentMethod( $card_id );

    		if( $log ) {
	    		$activity_data = array(
					'customer_id' => $this->customer->getId(),
					'name'        => $this->customer->getFirstName() . ' ' . $this->customer->getLastName(),
					'old_card'    => $oldCard,
					'new_card'    => $newCard,
				);

				$this->model_account_activity->addActivity( 'default_card', $activity_data);
    		}

    		$ret['success'] = $this->a->__( 'Default payment method has been changed' );

		} catch ( \Exception $e ) {
			if( $log ) {
				$activity_data = array(
					'customer_id' => $this->customer->getId(),
					'name'        => $this->customer->getFirstName() . ' ' . $this->customer->getLastName(),
					'card'        => $newCard,
					'message'     => $e->getMessage(),
				);

				$this->model_account_activity->addActivity( 'fail_default_card', $activity_data);
			}

			$ret['error'] = $e->getMessage();
			$this->a->error( $e );
		}

		$this->response->setOutput( json_encode( $ret ) );
	}

	/**
	 * Pay in one click button contents
	 * @param int $product_id
	 * @return string
	 * @throws \Advertikon\Exception
	 * @throws Exception
	 */
	public function pay_button( $product_id ) {
	    try {
            $button = new \Advertikon\Stripe\Button();
            $button->setProductId( $product_id );

            if ( !$button->doShow() ) {
                return '';
            }

            $this->document->addStyle( $this->a->u()->catalog_css( 'advertikon/animate.min.css' ) );
            $this->document->addStyle( $this->a->u()->catalog_css( 'advertikon/advertikon.css' ) );
            $this->document->addStyle( $this->a->u()->catalog_css( 'advertikon/stripe/form.css' ) );

            $data = $button->get();
            $this->response->addHeader( 'Content-Type: text/html' );
            $this->a->setApiKey();

            return $this->a->load->view( $this->a->get_template( $this->a->type . '/advertikon/stripe/button' ), $data );

        } catch ( Exception $e ) {
	        $this->a->error( $e );
        }

        return '';
	}

	public function charge_success() {
        try {
            if ( $this->a->requestParam( 'isPayButton' ) ) {
                $button = new Button();
                $button->resetCart();
            }

        } catch ( Exception $e ) {
            $this->a->error( $e );
        }
    }

    /**
     * @throws \Advertikon\Exception
     */
	public function create_order() {
        $button = new \Advertikon\Stripe\Button( $this->a->requestParam() );
		$this->response->setOutput( json_encode( $button->addOrder() ) );
	}

    /**
     * @throws \Advertikon\Exception
     * @throws \Advertikon\Stripe\Exception
     */
	public function button_get_price() {
	    $button = new \Advertikon\Stripe\Button( $this->a->requestParam() );
		$this->response->setOutput( json_encode( $button->getPrice() ) );
	}

	/**
	 * Web-hooks action
	 * @return void
	 */
	public function webhooks() {
		$out = '';

		try {
			header( 'HTTP/1.0 200 OK' );
			$evt = $this->model->getEvent();
			$ignore_store_hash = false;

			$this->a->log( "Event {$evt->type} received" );

			if ( !in_array( $evt->type, Advertikon::EVENTS ) ) {
				die;
			}

			if ( !in_array( $evt->type, [ 'customer.updated' ] ) ) {
                $this->a->log( $evt );
            }

            $ignore_store_hash = strpos( $evt->type, 'invoice.' ) === 0 || count( Store::get_stores() ) === 0;

			if ( !$ignore_store_hash && !$evt->data->object->metadata->store_hash ) {
				throw new Exception( 'Store hash is missing' );
			}

			if ( !$ignore_store_hash && is_null( $this->a->checkStoreHash( $evt->data->object->metadata->store_hash ) ) ) {
				throw new Exception( 'Different store' );
			}

			switch( $evt->type ) {
				case 'charge.succeeded' :
					$this->model->onCharge( $evt->data->object );
					break;
				case 'charge.captured':
				case 'charge.refunded':
					$this->model->onOrderChangeStatus( $evt->data->object );
					//
					break;
				case 'source.chargeable':
					//$this->model->remember_source( $evt->data->object );
					$this->model->doCharge( $evt->data->object );
					//$this->model->charge_no_charge( $evt->data->object );
					break;
				case 'source.failed':
					//$this->model->remember_source( $evt->data->object );
					break;
				case 'customer.subscription.created' :
                case 'customer.subscription.deleted' :
					$this->model->subscriptionChangeStatus( $evt->data->object );
					$this->model->callback( $evt );
					break;
				case 'customer.subscription.updated' :
					$this->model->subscriptionChangeStatus( $evt->data->object );
					break;
				case 'customer.deleted':
				case 'customer.updated':
					$this->model->customer_update( $evt->data->object );
					break;
				case 'invoice.upcoming':
//					$this->model->subscription_invoice_upcoming( $evt->data->object ); // check last cycle
					break;
				case 'invoice.payment_succeeded' :
				case 'invoice.payment_failed' :
					//$this->model->subscription_invoice_upcoming( $evt->data->object ); // check last cycle
					$this->model->subscriptionOnPay( $evt->data->object );
					break;
			}

		} catch( Throwable $e ) {
			$this->a->error( $e );
		}

		echo $out;
	}

	public function log() {
		$this->a->console->tail();
	}

	/**
	 * Buttons to manage recurring plan for customer's account page
	 * @return String
	 * @throws \Advertikon\Exception
	 */
	public function recurringButtons() {
        if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
            return '';
        }

        $this->document->addStyle( $this->a->u()->catalog_css( 'advertikon/advertikon.css' ) );

        $data  = [];
        $locale = [
            'cancelSubscriptionText' => $this->a-> __( "Cancel subscription?" ),
            'order_recurring_id'     => $this->a->requestParam( 'order_recurring_id',
                $this->a->requestParam( 'recurring_id') )
        ];

        $data['requireJs']  = $this->a->requireJs(['stripe/recurring_buttons' => 'buttons' ], [], $locale );
        $data['can_cancel'] = Setting::get( 'edit_subscription', $this->a );

        $data['cancel_now_button']   = \Advertikon\Setting::get( 'cancel_subscription_now', $this->a ) ?
            (new \Advertikon\Element\Bootstrap\Button())
                ->isDefault()
                ->url( $this->a->u()->url( 'recurring_delete_now' ) )
                ->title( $this->a->__( 'Cancel immediately' ) )
                ->icon('fa-times')
                ->id('adk-cancel-now')
            : '';

        $data['cancel_button'] = (new \Advertikon\Element\Bootstrap\Button())
            ->isDefault()
            ->url( $this->a->u()->url( 'recurring_delete' ) )
            ->title( $this->a->__( 'Cancel at period end' ))
            ->icon('fa-calendar')
            ->id('adk-cancel-period');

        $data['refresh_button'] = (new \Advertikon\Element\Bootstrap\Button())
            ->isDefault()
            ->url( $this->a->u()->url( 'recurring_refresh' ) )
            ->title( $this->a-> __( 'Refresh' ))
            ->icon('fa-refresh')
            ->id('adk-refresh');

        return $this->a->load->view(
            $this->a->get_template( $this->a->type . '/advertikon/stripe/recurring_button' ),
            $data
        );
	}

    /**
     * Update recurring order
     * @return void
     */
    public function recurring_refresh() {
        $ret = [];

        try {
            if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
                return;
            }

            $orderId = $this->a->requestParam( 'order_recurring_id' );

            if ( is_null( $orderId ) ) {
                throw new \Advertikon\Exception( 'Recurring order ID is missing' );
            }

            $aRecurring = \Advertikon\Stripe\Recurring::get( $orderId );

            if ( !$aRecurring->exists() ) {
                throw new \Advertikon\Exception( $this->a->__( 'Order is missing' ) );
            }

            $subscription = $aRecurring->fetch();

            if ( is_null( $subscription ) ) {
                throw new Exception( 'Subscription is missing' );
            }

            $aRecurring->updateStatus( $subscription );
            $this->request->get['order_recurring_id'] = $orderId;

            if ( $this->a->is_admin() ) {
                $this->load->controller( 'sale/recurring/info' );

            } else {
                $this->load->controller( 'account/recurring/info' );
            }

            $ret['html'] = $this->response->getOutput();
            $ret['success'] = 'ok';

        } catch( \Advertikon\Exception $e ) {
            $ret['error'] = $e->getMessage();
            $this->a->error( $e );

        } catch( \Exception $e ) {
            $ret['error'] = 'Script error';
            $this->a->error( $e );
        }

        echo( json_encode( $ret ) );
        die;
    }

    /**
     * Cancel subscription at the next period end
     * @return void
     */
    public function recurring_delete() {
        if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
            return;
        }

        $ret = [];

        try {
            if ( !isset( $this->request->request['order_recurring_id'] ) ) {
                throw new \Advertikon\Exception( 'Recurring order ID is missing' );
            }

            $orderId = $this->a->requestParam( 'order_recurring_id' );

            if ( is_null( $orderId ) ) {
                throw new \Advertikon\Exception( 'Recurring order ID is missing' );
            }

            $aRecurring = \Advertikon\Stripe\Recurring::get( $orderId );

            if ( !$aRecurring->exists() ) {
                throw new \Advertikon\Exception( $this->a->__( 'Order is missing' ) );
            }

            $aRecurring->cancelAtPeriodEnd();
            $ret['success'] = $this->a->__('Subscription cancellation has been scheduled at the next period end' );

        } catch( \Advertikon\Exception $e ) {
            $ret['error'] = $e->getMessage();
            $this->a->error( $e );

        } catch( \Exception $e ) {
            $ret['error'] = 'Script error';
            $this->a->error( $e );
        }

        $this->a->json_response( $ret );
    }

    /**
     * Cancel immediately recurring
     * @return void
     */
    public function recurring_delete_now() {
        if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
            return;
        }

        $ret = [];

        try {
            if ( !isset( $this->request->request['order_recurring_id'] ) ) {
                throw new \Advertikon\Exception( 'Recurring order ID is missing' );
            }

            $orderId = $this->a->requestParam( 'order_recurring_id' );

            if ( is_null( $orderId ) ) {
                throw new \Advertikon\Exception( 'Recurring order ID is missing' );
            }

            $aRecurring = \Advertikon\Stripe\Recurring::get( $orderId );

            if ( !$aRecurring->exists() ) {
                throw new \Advertikon\Exception( $this->a->__( 'Order is missing' ) );
            }

            $aRecurring->cancel();
            $ret['success'] = $this->a->__('Subscription has been cancelled' );

        } catch( \Advertikon\Exception $e ) {
            $ret['error'] = $e->getMessage();
            $this->a->error( $e );

        } catch( \Exception $e ) {
            $ret['error'] = 'Script error';
            $this->a->error( $e );
        }

        $this->a->json_response( $ret );
    }

//	public function check_payment() {
//		$data = [];
//		$data['key'] = $this->a->get_public_key();
//		$data['source'] = $this->a->request( 'source' );
//		$data['secret'] = $this->a->request( 'client_secret' );
//		$locale = $this->get_locale_data();
//		$locale['paymentMethod'] = $this->a->request( 'method' );
//		$locale['productId'] = $this->a->request( 'product_id' );
//		$data['locale'] = json_encode( $locale );
//
//		$this->response->setOutput( $this->a->load->view( $this->a->get_template( $this->a->type . '/advertikon/stripe/check_payment' ) , $data ) );
//	}
//
//	public function source_redirect() {
//		$ret = [ 'type' => 'source_authorization' ];
//		$source = isset( $this->request->get['source'] ) ? $this->request->get['source'] : null;
//		$intent = $this->a->session_get( 'adk_payment_intent_id' );
//		$is_source = $this->a->g( 'is_source' );
//
//		try {
//			if ( !$source && !$intent ) {
//				throw new Exception( $this->a->__( 'Processing error' ) );
//			}
//
//			if ( !$is_source ) {
//				$this->a->update_intent( $intent, [ 'source' => $source ] );
//			}
//
//			$ret['source'] = \Stripe\Source::retrieve( $source );
//			$ret['success'] = 'ok';
//
//		} catch ( Exception $e ) {
//			$ret['error'] = $e->getMessage();
//			$this->a->error( $e );
//		}
//
//		$this->response->setOutput( "<script>window.parent.postMessage( '" . addslashes( json_encode( $ret ) ) . "', '*' );</script>" );
//	}
//
//	public function fetch_source() {
//		$ret = [];
//		$id = $this->a->g( 'id' );
//		$meta_id = $this->a->g( 'meta_id' );
//
//		try {
//			if ( !$id ) {
//				throw new Exception( $this->a->__( "Source ID is missing" ) );
//			}
//
//			if ( !$meta_id ) {
//				throw new Exception( $this->a->__( "Session ID is missing" ) );
//			}
//
//			/** @var Stripe\Source $source */
//			$source = \Stripe\Source::retrieve( $id );
//
//			// Hack
//			$source->metadata->no_charge = true;
//			$source->metadata->meta_id = $meta_id;
//			$source->currency = $this->session->data['adk_currency'];
//
//			$this->model->charge_no_charge( $source, true );
//			$ret['success'] = 'ok';
//			$ret['source'] = $source->__toArray( true );
//
//		} catch ( Exception $e ) {
//			$ret['error'] = $e->getMessage();
//			$this->a->error( $e );
//		}
//
//		$this->response->setOutput( json_encode( $ret ) );
//	}
//
//	public function sync_source() {
//		$ret = [];
//		$meta_id = $this->a->g( 'meta_id' );
//
//		try {
//			if ( !$meta_id ) {
//				throw new Exception( $this->a->__( "Session ID is missing" ) );
//			}
//
//			$status = $this->model->check_sync_source( $meta_id );
//
//			if ( !is_null( $status ) ) {
//				if ( $status ) {
//					$ret['success'] = 'ok';
//
//				} else {
//					$ret['error'] = $this->a->__( 'Payment is failed' );
//				}
//			}
//
//		} catch ( Exception $e ) {
//			$ret['error'] = $e->getMessage();
//			$this->a->error( $e );
//		}
//
//		$this->response->setOutput( json_encode( $ret ) );
//	}
//
//	public function window_close() {
//		echo '<script>window.close();</script>';
//		die;
//	}

    /**
     * Emails list of renewed subscriptions
     * @throws \Advertikon\Exception
     * @throws Exception
     */
    public function emailUpdates() {
        $debug = true;

        if ( $debug ) {
            echo "Sending subscription renew summary\n";
        }

        $this->a->log( "Sending statement of daily renewed subscriptions" );

        $format = "M j, y";
        $today = date( $format );
        $subscriptions = $this->a->fetch_api_subscription_all()->data;
        $header = "List of renewed subscriptions ($today):\n\n\n";
        $bodies = [];
        $to = $this->a->config->get('config_email');
        $subject = 'Renewed subscriptions daily report';

        if ( count( $subscriptions ) == 0 ) {
            $this->a->log( "No active subscriptions. Exit" );
            return;
        }

        /** @var Stripe\Subscription $subscription */
        foreach( $subscriptions as $subscription ) {
            $body = '';
            $dateRenewed = date( $format, $subscription->current_period_start );

            if ( $debug ) {
                echo sprintf(
                    "Subscription: %s. Current period start: %s, Current period end: %s\n",
                    $subscription->plan->nickname ?: $subscription->plan->product,
                    date( $format, $subscription->current_period_start ),
                    date( $format, $subscription->current_period_end )
                );
            }

            if ( $dateRenewed !== $today ) {
                echo "Doesn't match. Skip it\n";
                continue;
            }

            $recurringOrderId = $subscription->metadata['recurring_order_id'];

            if ( $recurringOrderId ) {
                $q = $this->a->q()->log( 1 )->run_query( [
                    'table' => 'order_recurring',
                    'where' => [
                        'field'     => 'order_recurring_id',
                        'operation' => '=',
                        'value'     => $recurringOrderId
                    ],
                    'join' => [
                        'table' => ['o' => 'order' ],
                        'using' => 'order_id'
                    ]
                ] );

                $body .= sprintf(
                    "Subscription: %s\nCustomer: %s %s\nOrder ID: %s\nRecurring Order ID: %s\n",
                    $q['recurring_description'],
                    $q['firstname'],
                    $q['lastname'],
                    $q['order_id'],
                    $q['order_recurring_id']
                );

            } else {
                $body .= sprintf( "Subscription name: %s\n",
                   $subscription->plan->nickname ?: $subscription->plan->product
                );
            }

            $body .= sprintf( "Created: %s\nCurrent period: %s - %s\n",
                date( $format, $subscription->created ),
                date( $format, $subscription->current_period_start ),
                date( $format, $subscription->current_period_end )
            );

            $bodies[] = $body;
        }

        if ( $bodies ) {
            $text = $header . implode( str_repeat("*", 40 ) . "\n\n", $bodies );

            $mail = new Mail();
            $mail->setFrom( $to );
            $mail->setTo( $to );
            $mail->setSubject( $subject );
            $mail->setReplyTo( $to );
            $mail->setText( $text );
            $mail->setSender( $this->a->config->get( 'config_name' ) );
            $mail->send();

            if ( $debug ) {
                echo sprintf( "Sent email to %s with content:\n %s", $to, $text );
            }
        }
    }


    /**
     * @throws \Advertikon\Exception
     * @throws \Stripe\Error\Api
     * @throws Exception
     */
    public function delete_subscription_all() {
        throw new Exception('Error');
        $this->a->setApiKey();
        $list = \Stripe\Subscription::all(['limit'=>100]);

        /** @var \Stripe\Subscription $subscription */
        foreach( $list->data as $subscription ) {
            $subscription->cancel();
            echo 'Cancel ' . $subscription->id . '<br>';
        }
    }
}
