<?php
/**
  * Catalog Advertikon Stripe Model
  *
  * @author Advertikon
  * @package Stripe
  * @version 5.0.44     
  */

use Advertikon\Element\Span;
use Advertikon\Setting;
use Advertikon\Stripe\Order;
use Advertikon\Stripe\OrderPrice;
use Advertikon\Stripe\PaymentOption;
use Advertikon\Stripe\PaymentOption\Card;
use Advertikon\Stripe\PaymentOption\Vendor;
use Advertikon\Stripe\Recurring;
use Advertikon\Stripe\StripeAccount;

/**
 * Class ModelExtensionPaymentAdvertikonStripe
 * @property DB $db
 * @property Request $request
 * @property Response $response
 * @property Config $config
 * @property Document $document
 * @property Loader $load
 * @property Language $language
 * @property Session $session
 * @property Url $url
 * @property \Cart\Customer $customer
 * @property \Cart\Cart $cart
 * @property \Cart\Currency $currency
 * @property Log $log
 *
 * @property ModelAccountCustomer $model_account_customer
 * @property ModelExtensionExtension $model_extension_extension
 * @property ModelSettingExtension $model_setting_extension
 * @property ModelCatalogProduct $model_catalog_product
 * @property ModelCheckoutOrder $model_checkout_order
 */
class ModelExtensionPaymentAdvertikonStripe extends Model {

	/** @var \Advertikon\Stripe\Advertikon|null */
	public $a = null;
	public $payment_type = '';

	/**
	 * ModelExtensionPaymentAdvertikonStripe constructor.
	 * @param $registry
	 * @throws \Advertikon\Exception
	 */
	public function __construct($registry ) {
		parent::__construct( $registry );
		$this->a = Advertikon\Stripe\Advertikon::instance();
	}

	/**
	 * @param $address
	 * @param $total
	 * @return array|bool
	 * @throws \Advertikon\Exception
	 * @throws Exception
	 */
	public function getMethod( $address, $total ) {
		try{
			if ( !$this->a->is_modification_ok() ) {
				throw new Advertikon\Stripe\Exception( 'Extension\'s modification has not been applied' );
			}
						
			if ( Setting::get( 'recurring_only', $this->a ) && !$this->cart->hasRecurringProducts() ) {
				throw new Advertikon\Stripe\Exception( 'Cart doesn\'t contain recurring product' );
			}

			// Allowed Geo Zones
			$stripe_geo_zones = Setting::get( 'geo_zone', $this->a );

			if ( !is_array( $stripe_geo_zones ) ) {
				$stripe_geo_zones = $this->a->object_to_array( $stripe_geo_zones );
			}

			// -1 - all geo-zones
			if ( !empty( $stripe_geo_zones ) && !in_array( '-1', $stripe_geo_zones ) ) {
				$zones = $this->db->query(
					"SELECT `geo_zone_id` FROM `" . DB_PREFIX . "zone_to_geo_zone`
					WHERE `country_id` = " . (int)$address['country_id'] . "
					AND (`zone_id` = " . (int)$address['zone_id'] . " OR `zone_id` = 0)"
				);

				$z = array();

				// Collect all geo-zones' ID in one-dimensional array
				if ( isset( $zones->rows ) ) {
					foreach( $zones->rows as $zz ) {
						$z[] = $zz['geo_zone_id'];
					}
				}

				if ( !array_intersect( $z, (array)$stripe_geo_zones ) ) {
					throw new Advertikon\Stripe\Exception( 'Forbidden payment Geo-Zone' );
				}
			}

			// Allowed stores
			$stores = Setting::get( 'stores', $this->a, [] );

			if ( !is_array( $stores ) ) {
				$stores = $this->a->object_to_array( $stores );
			}

			// -1 - all stores
			if ( !empty( $stores ) && !in_array( '-1', $stores ) ) {
				if( !in_array( $this->config->get( 'config_store_id' ), $stores ) ) {
					throw new Advertikon\Stripe\Exception( 'Forbidden store' );
				}
			}

			// Allowed customer groups
			$stripe_groups = Setting::get( 'customer_groups', $this->a, ['0'] );

			if ( !is_array( $stripe_groups ) ) {
				$stripe_groups = $this->a->object_to_array( $stripe_groups );
			}

			if( 
				(
					$this->customer->isLogged() &&
					!array_intersect( array( 0, '0', $this->customer->getGroupId() ), $stripe_groups )
				) ||
				(
					!$this->customer->isLogged() &&
					!array_intersect( array( 0, '0', $this->config->get( 'config_customer_group_id' ) ), $stripe_groups )
				)
			) {
				throw new Advertikon\Stripe\Exception( 'Forbidden customer group' );
			}

			$min_total_setting = Setting::get( 'total_min', $this->a );
			$min_total = $this->currency->convert(
				(float)$min_total_setting,
				$this->config->get( 'config_currency' ),
				$this->session->data['currency' ]
			);

			$max_total_setting = Setting::get( 'total_max', $this->a );
			$max_total = $this->currency->convert(
				(float)$max_total_setting,
				$this->config->get( 'config_currency' ),
				$this->session->data['currency' ]
			);

			// Min total amount
			if ( is_numeric( $min_total_setting ) && $total < (float)$min_total && !$this->cart->hasRecurringProducts() ) {
				throw new Advertikon\Stripe\Exception(
					sprintf(
						'The order total amount is: %s, minimum permitted total amount is: %s',
						$this->currency->format( $total, $this->session->data['currency'], 1 ),
						$this->currency->format( $min_total, $this->session->data['currency'], 1 )
					)
				);
			}

			// Max total amount
			if ( is_numeric( $max_total_setting ) && $total > (float)$max_total ) {
				throw new Advertikon\Stripe\Exception(
					sprintf(
						'The order total amount is: %s, maximum permitted total amount is: %s',
						$this->currency->format( $total, $this->session->data['currency'], 1 ),
						$this->currency->format( $max_total, $this->session->data['currency'], 1 )
					)
				);
			}

		} catch( Advertikon\Stripe\Exception $e ) {
			$message = sprintf( 'Stripe Gateway disabled. Reason: "%s"', $e->getMessage() );
			$this->a->error( $message );
			$this->log->write( $message );

			return false;

		} catch ( Advertikon\Exception $e ) {
			$this->a->error( sprintf( 'Stripe Gateway disabled. Reason: script error' )	);
			$this->a->error( $e );
			$this->log->write( sprintf( 'Stripe Gateway disabled. Reason: "%s"', $e->getMessage() ) );

			return false;
		}

		$terms = (new Span())->style()->padding('3px')->stop()->verticalAlign('sub')->stop();

		/** @var PaymentOption $option */
		foreach( PaymentOption::getAllEnabled() as $option ) {
			if ( !$option->isShowIcon() || ( $this->cart->hasRecurringProducts() && !$option->supportsRecurring() ) ) continue;

			if ( $option->isCard() ) {
				/** @var Vendor $vendor */
				foreach ( $option->enabledVendors() as $vendor ) {
					$terms->children( (new \Advertikon\Element\Image( $vendor->getIcon() ))
						->style()->height('25px!important')->margin('2px')->stop()->stop() );
				}

			} else {
				$terms->children( (new \Advertikon\Element\Image( $option->iconUrl() ))
					->style()->height('25px!important')->margin('2px')->stop()->stop() );
			}
		}

		return array(
			'code'       => $this->a->code,
			'title'      => Setting::get( 'test_mode', $this->a ) ?
				$this->a->__( 'caption_sandbox_title' ) : $this->a->__( 'caption_title' ),
			'sort_order' => Setting::get( 'sort_order', $this->a ),
			'terms'      => $terms->isEmpty() ? '' : $terms . '',
		);
	}

	/**
	 * Declare whether extension support recurring payments
	 * @return boolean
	 */
	public function recurringPayments() {
		try {
			if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
				return false;
			}

			if ( Setting::get( 'recurring_only_logged', $this->a ) && !$this->customer->getId() ) {
				throw new Advertikon\Stripe\Exception( 'Customer is not logged in' );
			}

			foreach( $this->cart->getProducts() as $product ) {
				if ( !empty($product['recurring']) && OrderPrice::isTrial( $product['recurring'] ) &&
					!OrderPrice::isSupportedRecurring( $product['recurring'] ) ) {

					throw new Advertikon\Stripe\Exception(
						sprintf(
							'Stripe doesn\'t allows non-free trial for subscriptions (product "%s" has profile "%s" with non-free trial period: %s). ' .
							'In order to set a non-free trial period set the one-off price of the product to the needed amount',
							$product['name'],
							$product['recurring']['name'],
							$this->currency->format( $product['recurring']['trial_price'], $this->config->get( 'config_currency') )
						)
					);
				}
			}

		} catch( Advertikon\Stripe\Exception $e ) {
			$this->a->error(
				sprintf( '[Stripe] Unable to handle recurring payment: %s', $e->getMessage() )
			);

			return false;
		}

		return true;
	}

	/**
	 * Subscription payment failure callback
	 * @param \Stripe\Invoice $invoice Stripe invoice
	 * @throws \Advertikon\Exception on system error
	 */
	public function subscriptionOnPay( \Stripe\Invoice $invoice ) {
		if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
			return;
		}

		if ( $invoice->subscription ) {
            $aRecurring = Recurring::getBySubscriptionId( $invoice->subscription );

            if ( !$aRecurring->exists() ) {
                throw new Advertikon\Exception ( "No such subscription: {$invoice->subscription}" );
            }

            $aRecurring->addTransaction( $invoice );
		}
	}

	/**
	 * @return \Stripe\Event|null
	 * @throws \Advertikon\Exception
	 */
	public function getEvent() {
		$payload = @file_get_contents( 'php://input' );
		$event = null;
		$accounts = StripeAccount::all();

		if ( count( $accounts ) === 1 ) {
			return $this->webhookNoSecret( $payload );
		}

		/** @var StripeAccount $account */
		foreach( $accounts as $account ) {
			$this->a->log( 'Trying account ' . $account->getAccountName() );
			$s = [ $account->getLiveSigningKey(),  $account->getTestSigningKey() ];

			foreach( $s as $secret ) {
				if ( !$secret ) {
					continue;
				}

				$this->a->log( 'Trying key ' . $this->a->obscure_str( $secret ) );
				$event = $this->webhookSecret( $payload, $secret );

				if ( $event ) {
				    $this->a->setAccount( $account->getCode() );
				    return $event;
                }

				$this->a->log( 'Wrong secret' );
			}

			$this->a->log( 'Wrong account' );
		}

		if ( !$event ) {
			$this->a->error( 'Failed to find out proper account' );
			http_response_code( 400 );
			exit();
		}

		return $event;
	}

    /**
     * @param $payload
     * @return \Stripe\Event
     * @throws \Advertikon\Exception
     * @throws \Advertikon\Stripe\Exception
     */
	private function webhookNoSecret( $payload ) {
		try {
		    $this->a->setAccount( StripeAccount::byCurrency()->getCode() );
			return \Stripe\Event::constructFrom( json_decode( $payload, true) );

		} catch( UnexpectedValueException $e ) {
			$this->a->error( $e );
			http_response_code(400);
			exit();
		}
	}

    /**
     * @param $payload
     * @param $secret
     * @return \Stripe\Event|null
     */
	private function webhookSecret( $payload, $secret ) {
		$sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];

        try {
			return \Stripe\Webhook::constructEvent( $payload, $sig_header, $secret );

		} catch( UnexpectedValueException $e) {
			http_response_code(400);
			exit();
		} catch(\Stripe\Error\SignatureVerification $e) {}

		return null;
	}

    /**
     * Change subscription status in response to web-hook
     * @param \Stripe\Subscription $subscr
     * @return void
     * @throws \Advertikon\Exception
     * @throws Exception
     */
	public function subscriptionChangeStatus( \Stripe\Subscription $subscr ) {
		if ( !class_exists( 'Advertikon\Stripe\Recurring' ) ) {
			return;
		}

		$recurring = Recurring::getBySubscriptionId( $subscr->id );

		if ( !$recurring->exists() ) {
		    throw new Exception( 'No subscription with ID' . $subscr->id );
        }

		$recurring->updateStatus( $subscr );
	}

	/**
	 * Customer was deleted notification
	 * @param \Stripe\Customer $customer
	 * @return void
	 */
	public function customer_update( \Stripe\Customer $customer ) {
		if ( $this->a->do_cache ) {
			$this->a->cache->delete( $customer->id );
			$this->a->cache->delete( $customer->id . '_card_all' );
		}
	}

	/**
	 * Sends callback on subscription status change
	 * @param object $evt Stripe event object
	 * @return void
	 * @throws \Advertikon\Exception
	 */
	public function callback( $evt ) {
		if ( !$this->a->isExtended ) {
			return;
		}

		$post = [];

		if ( 'customer.subscription.created' === $evt->type ) {
			$url    = Setting::get( 'create_subscription_callback', $this->a );
			$data   = Setting::get( 'create_subscription_callback_data', $this->a );
			$status = 'new';

		} elseif ( 'customer.subscription.deleted' === $evt->type ) {
			$url    = Setting::get( 'cancel_subscription_callback', $this->a );
			$data   = Setting::get( 'cancel_subscription_callback_data', $this->a );
			$status = 'cancel';

		} else {
			$mess = 'Failed to send query to callback URL: subscription status is undefined';
			throw new \Advertikon\Exception( $mess );
		}

		if ( !$url ) {
			return;
		}

		$subscr = $evt->data->object;
		$aRecurring = Recurring::getBySubscriptionId( $subscr->id );

		if ( !$aRecurring->exists() ) {
			$mess = sprintf(
				'Failed to send query to callback URL: stripe subscription %s is not registered in OC store',
				$subscr->id
			);

			throw new \Advertikon\Exception( $mess );
		}

		$aCustomer = \Advertikon\Stripe\Customer::getByStripeId( $subscr->customer );

		if ( !$aCustomer->exists() ) {
			$mess = 'Failed to send query to callback URL: OC customer is undefined';
			throw new \Advertikon\Exception( $mess );
		}

		$oc_customer         = $aCustomer->ocId();
		$stripe_customer     = $subscr->customer;
		$oc_subscription     = $aRecurring->orderId();
		$stripe_subscription = $subscr->id;

		if ( $data ) {
			foreach( explode( ',', $data ) as $d1 ) {
				list( $name, $val ) = explode( '=', $d1, 2 );

				if( !( $name = trim( $name ) ) ) {
					continue;
				}

				$post[ urlencode( $name ) ] = urlencode( trim( $val ) );
			}
		}

		$email = '';

		try {
			$customer = \Stripe\Customer::retrieve( $subscr->customer );
			$email = $customer->email;

		} catch ( \Exception $e ) {
			$this->a->error( $e );
		}

		$post['oc_customer']         = urlencode( $oc_customer );
		$post['stripe_customer']     = urlencode( $stripe_customer );
		$post['oc_subscription']     = urlencode( $oc_subscription );
		$post['stripe_subscription'] = urlencode( $stripe_subscription );
		$post['status']              = urlencode( $status );
		$post['email']               = urlencode( $email );

		$fd = fopen( 'php://temp', 'w' );
		$ch = curl_init();
		curl_setopt( $ch, CURLOPT_URL, $url );
		curl_setopt( $ch, CURLOPT_HEADER, 0 );
		curl_setopt( $ch, CURLOPT_POST, 1 );
		curl_setopt( $ch, CURLOPT_POSTFIELDS, $post );
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
		curl_setopt( $ch, CURLOPT_VERBOSE, 1 );
		curl_setopt( $ch, CURLOPT_STDERR, $fd );
		curl_exec( $ch );

		if( curl_errno( $ch ) ) {
			$mess = 'CURL error: ' . curl_error( $ch );
			rewind( $fd );
			$this->a->error( $mess . ': ' . fread( $fd, 4096 ) );
		}

		curl_close( $ch );
		fclose( $fd );
	}

	/**
	 * @param \Stripe\Charge $charge
	 * @throws Exception
	 */
	public function onCharge( \Stripe\Charge $charge ) {
		if ( Order::is_processed( $charge->id ) ) {
			throw new Exception( 'Charge ' . $charge->id . ' has been processed already' );
		}

		$this->a->log( 'Charge start' );

		if ( Advertikon\Stripe\Advertikon::PARANOID_MODE ) {
			$charge = \Stripe\Charge::retrieve( $charge->id );
		}

		$orderId = isset( $this->session->data['order_id'] ) ? $this->session->data['order_id'] : $charge->metadata['order_id'];

		if ( !$orderId ) {
			throw new Exception( 'Order ID is missing' );
		}

		$order = \Advertikon\Stripe\Order::create( $orderId, $charge->id );

		if ( $charge->captured ) {
			$order->setStatusCapture();

		} else {
			$order->setStatusAuthorize();
		}

		$this->a->log( 'Charge end' );
	}

	/**
	 * @param \Stripe\Charge $charge
	 * @throws \Advertikon\Exception
	 */
	public function onOrderChangeStatus( \Stripe\Charge $charge ) {
		$this->a->log( 'Order change status' );

		if ( Advertikon\Stripe\Advertikon::PARANOID_MODE ) {
			$charge = \Stripe\Charge::retrieve( $charge->id );
		}

		$order = Order::getByCode( $charge->id );

		if ( !$order->exists() ) {
			throw new Advertikon\Exception( "No corresponding order" );
		}

		if ( $charge->refunded ){
			$order->setStatusRefund();

		} else if ( $charge->captured ) {
			$order->setStatusCapture();
		}
	}

	/**
	 * Charge is source depending on conditions
	 * @param \Stripe\Source $source
	 */
	public function doCharge( \Stripe\Source $source ) {
        $this->a->log( 'Charging the source...' );

        $charge = \Stripe\Charge::create( [
            'amount'      => $source->amount,
            'currency'    => $source->currency,
            'capture'     => true,
            'source'      => $source->id,
            'metadata'    => $source->metadata->toArray()
        ] );

        $this->a->log( $charge );
    }
}
