define( "stripe/payment/card", ["stripe/card-preview"], function( Preview ) {
    function Card( loc, parent ) {
        var locale = loc;
        var card;
        var expiry;
        var cvv;
        var $cardField = $( "#p-card" );
        var $cardNumberField = $( "#card-number" );
        var $cardExpiryField = $( "#card-expiry" );
        var $cardCvvField = $( "#card-cvv" );
        var $savedCardField = $( "#saved-card" );
        var $rememberMeField = $( "#remember-me" );
        var elements;
        var stripe;
        var secret;
        var payment = parent;
        var recurringCartIds = loc.recurringIds;
        var method;

        this.pay = function() {
            nextStep();
        };

        function nextStep() {
            if ( localStorage.getItem('confirmSubscription') ) {
                console.log("reconfirm subscription");
                reConfirmSubscription();
                return;
            }

            $.getJSON( locale.stepUrl + '&t=' + new Date().getTime() )
                .fail( function( resp ) {
                    console.error( resp );
                    payment.error( locale.scriptErrorMsg )
                } )
                .done( function ( resp ) {
                    if ( resp.error ) {
                        payment.error( resp.error );

                    } else if ( resp.step ) {
                        console.log( resp.step);
                        switch( resp.step ) {
                            case 'subscription':
                                recurringPayment();
                                break;
                            case 'one-time':
                                payOneTime();
                                break;
                            case 'finish':
                                finish();
                                break;
                            case "retry":
                                retry();
                                break;
                            default:
                                payment.error( locale.scriptErrorMsg );
                                break;
                        }

                    } else {
                        payment.error( locale.scriptErrorMsg );
                    }
                } );
        }

        function setStep( cartId, invoiceId, oneTimeStatus ) {
            $.getJSON( locale.setStepUrl + '&t=' + new Date().getTime(), {
                cart_id:    cartId,
                invoice_id: invoiceId,
                one_time:   oneTimeStatus
            } )
                .fail( function( resp ) {
                    console.error( resp );
                    payment.error( locale.scriptErrorMsg )
                } )
                .done( function ( resp ) {
                    if ( resp.error ) {
                        payment.error( resp.error );

                    } else if ( resp.success ) {
                        if ( !invoiceId && oneTimeStatus !== '0' ) {
                            nextStep();
                        }

                    } else {
                        payment.error( locale.scriptErrorMsg );
                    }
                } );
        }

        function finish() {
           payment.finish();
        }

        function payOneTime() {
            if ( secret ) {
                confirmIntent( secret );

            } else {
                createIntent()
                    .done( function( resp ) {
                        if ( resp.error ) {
                            payment.error( resp.error );

                        } else if ( resp.secret ) {
                            secret = resp.secret;
                            $rememberMeField.attr("disabled", "disabled" );
                            confirmIntent( resp.secret );

                        } else {
                            payment.error( locale.scriptErrorMsg );
                        }
                    } )
                    .fail( function() { payment.error( locale.scriptErrorMsg ) } );
            }
        }

        function recurringPayment() {
            if ( !locale.stripeCustomer ) {
                createCustomer();

            } else {
                if ( !$savedCardField.val() && !method ) {
                    createPaymentMethod();

                } else {
                    createSubscription( method ? method : $savedCardField.val() );
                }
            }
        }

        function createIntent() {
            return $.post( locale.createIntentUrl, {
                saveCustomer: $rememberMeField.is( ":checked" ) ? "1" : "0",
            }, null, "json" );
        }

        function confirmIntent( sec ) {
            stripe.confirmCardPayment( sec, {
                payment_method: $savedCardField.val() ? $savedCardField.val() : {
                    card: card,
                    billing_details: payment.getOwner()
                }

            } ).then( function( result ) {
                if (result.error) {
                    payment.error( result.error.message );
                    setStep( null, null, '0' );

                } else {
                    if ( result.paymentIntent.status === 'succeeded' || result.paymentIntent.status === 'requires_capture' ) {
                        secret = null;
                        setStep(null, null, '1' );
                    }
                }
            } );
        }

        function createCustomer() {
            $.post( locale.createCustomerUrl, null, null, "json" )
                .fail( function( resp ) {
                    payment.error( locale.scriptErrorMsg );
                    console.error( resp );
                } )
                .done( function( resp ) {
                    if ( resp.error ) {
                        payment.error( resp.error );

                    } else if ( resp.customer ) {
                        locale.stripeCustomer = resp.customer;
                        createPaymentMethod();

                    } else {
                        payment.error( locale.scriptErrorMsg );
                    }
                } );
        }

        function createPaymentMethod( retry ) {
            return stripe.createPaymentMethod( {
                type: 'card',
                card: card,
            } )
            .then( function( result ) {
                if (result.error) {
                    payment.error( result.error.message );

                } else {
                    if ( retry ) {
                        retryInvoice( result.paymentMethod.id );

                    } else {
                        method = result.paymentMethod.id;

                        if ( localStorage.getItem('confirmSubscription' ) ) {
                            reConfirmSubscription();

                        } else {
                            createSubscription( result.paymentMethod.id );
                        }
                    }
                }
            } );
        }

        function createSubscription( paymentMethod ) {
            $.post( locale.createSubscriptionUrl, {
                paymentMethod: paymentMethod,
                customer:      locale.stripeCustomer
            }, null, "json" )
                .done( function( resp ) {
                    if ( resp.status === "succeeded" ) {
                        nextStep();

                    } else if ( resp.status === "requires_action" ) {
                        confirmSubscription( resp.secret, paymentMethod, resp.invoice_id, resp.cart_id );

                    } else if ( resp.status === 'requires_payment_method' ) {
                        payment.error( resp.error_message ? resp.error_message : locale.declinedCardMsg );
                        method = null;

                    } else if ( resp.error ) {
                        payment.error( resp.error );

                    } else {
                        payment.error( resp.error_message ? resp.error_message : locale.scriptErrorMsg );
                    }
                } )
                .fail( function( resp ) {
                    payment.error( locale.scriptErrorMsg );
                } );
        }

        function confirmSubscription( secret, paymentMethodId, invoiceId, cartId ) {
            return stripe.confirmCardPayment( secret, {
                    payment_method: paymentMethodId,
                } )
                .then( function (result) {
                    if (result.error) {
                        payment.error( result.error.message );
                        localStorage.setItem('confirmSubscription', secret );
                        localStorage.setItem('cartId', cartId );
                        method = null;

                    } else {
                        localStorage.removeItem('confirmSubscription');

                        if (result.paymentIntent.status === 'succeeded') {
                            setStep( cartId );
                        }
                    }
                } )
        }

        function retry() {
            if ( !$savedCardField.val() ) {
                createPaymentMethod( true );

            } else {
                retryInvoice( $savedCardField.val() );
            }
        }

        function retryInvoice( paymentMethod ) {
            $.post( locale.retryInvoiceUrl, {
                paymentMethod: paymentMethod,
                customer: locale.stripeCustomer
            }, null, 'json' )
                .fail( function ( resp ) {
                    console.error( resp );
                    payment.error( locale.scriptErrorMsg );
                } )
                .done( function ( resp ) {
                    if ( resp.error ) {
                        payment.error( resp.error );

                    } else {
                        nextStep();
                    }
                } );
        }

        function reConfirmSubscription() {
            if ( $savedCardField.val() || method ) {
                confirmSubscription(
                    localStorage.getItem('confirmSubscription'),
                    method ? method : $savedCardField.val(),
                    null,
                    localStorage.getItem("cartId") );

            } else {
                createPaymentMethod();
            }
        }

        function mountFields() {
            if ( card ) {
                card.mount( "#" + ( $cardField.length ? $cardField.attr( 'id' ) : $cardNumberField.attr( 'id' ) ) );
            }

            if ( expiry ) {
                expiry.mount( "#" + $cardExpiryField.attr( 'id' ) );
            }

             if ( cvv ) {
                 cvv.mount( "#" + $cardCvvField.attr( 'id' ) );
             }
        }

        function createFields() {
            if ( $cardField.length ) {
                card = elements.create( "card", {
                    hidePostalCode: true,
                    value:          {
                        postalCode: locale.zipCode
                    },
                    style: getElementStyle(),
                } );

                card.on( "change", cardOnChange );

            } else if ( $cardNumberField.length ) {
                card = elements.create( "cardNumber", {
                    style: getElementStyle(),
                } );

                card.on( "change", cardOnChange );
                card.on( "focus", function(){ Preview.number( true ) } );
                card.on( "blur", function(){ Preview.number() } );
            }

            if ( $cardExpiryField.length ) {
                expiry = elements.create( "cardExpiry", {
                    style: getElementStyle(),
                } );

                expiry.on( "focus", function(){ Preview.expiry( true ) } );
                expiry.on( "blur", function(){ Preview.expiry() } );
            }

            if ( $cardCvvField.length ) {
                cvv = elements.create( "cardCvc", {
                    style: getElementStyle(),
                });

                cvv.on("focus", function () { Preview.cvv(true) } );
                cvv.on("blur", function () { Preview.cvv() } );
            }
        }

        function getElementStyle() {
            return {
                base: {
                    color: locale.inputColor,
                    '::placeholder': {
                        color: locale.inputColor
                    },
                }
            };
        }

        function cardOnChange( msg ) {
            setVendorName( msg );
            checkVendor( msg );
        }

        var errorShown = false;

        function checkVendor( vendor ) {
            if ( vendor.brand !== "unknown" && -1 === locale.enabledVendors.indexOf(vendor.brand) ) {
                if ( !errorShown ) {
                    payment.error( locale.errorVendorMsg );
                    payment.disableButton();
                    errorShown = true;
                }

            } else {
                errorShown = false;
                payment.enableButton();
            }
        }

        function setVendorName( e ) {
            $( ".adk-cards-set-item" ).css( "filter", "grayscale(100%)" );
            $( ".adk-cards-set ." + e.brand ).css( "filter", "grayscale(0%)" );
        }

        +function init( self ) {
            stripe = Stripe( locale.pKey );
            elements = stripe.elements();
            createFields();
            mountFields();
            $(document).trigger("adk-request-layout");
        }( this );
    }

    return {
        init: function (locale, payment ) {
            return new Card( locale, payment );
        }
    }
} );