<?php
/**
 * Advertikon Stripe Class
 * @author Advertikon
 * @package Advertikon
 * @version 5.0.44  
 */

namespace Advertikon\Stripe;

use Advertikon\Setting;
use Advertikon\Sql;
use Advertikon\Store;

class Advertikon extends \Advertikon\Advertikon {
	// TODO: admin area Color input
	// TODO: smart button
	// TODO: check subscription end

	const STRIPE_API = '2020-03-02';
	const PARANOID_MODE = true;

	// ******************** Common properties ********************************//
	public $type = 'payment';
	public $code = 'advertikon_stripe';
	public $name = 'Advertikon Stripe';

	/* @var string Class instance indent */
	protected static $c = __NAMESPACE__;

	public $tables = array(
		'customer_table'         => 'advertikon_stripe_customer',
		'recurring_table'        => 'advertikon_stripe_recurring',
		'order'                  => 'advertikon_stripe_order',
		//'secret'                 => 'advertikon_stripe_secret',
	);
	// ******************* End of common properties **************************//

	public $version = '';  

	/**
	 * Payment methods
	 * @var integer
	 */	
	const PAYMENT_AUTHORIZE 		= 0;
	const PAYMENT_AUTHORIZE_CAPTURE	= 1;
	const PAYMENT_FRAUD_CHECK		= 2;

	/**
	 * Currency source for payment
	 * @var integer
	 */
	const CURRENCY_STORE = 1;
	const CURRENCY_ORDER = 2;

	const REFUND_REASON_DUPLICATE = 'duplicate';
	const REFUND_REASON_CUSTOMER  = 'requested_by_customer';
	const REFUND_REASON_FRAUD     = 'fraudulent';

	/**
	 * Maximum count of requests to Stripe Server
	 * @var integer
	 */
	const MAX_REQUEST_COUNT = 10;

	/**
	 * Default currency code for Stripe gateway
	 * @var string
	 */
	const DEFAULT_CURRENCY_CODE = 'USD';

	/**
	 * Minimum total amount for the Gateway
	 * @var float
	 */
	const MIN_AMOUNT = 0.5;

	/**
	 * @var String $secretKey Secret key
	 */
	protected $secret_key = null;

	/**
	 * @var String $pablicKey Public key
	 */
	protected $public_key = null;

	/**
	 * @var String $account Current account name
	 */
	protected $account = null;
	
	const FORM_IMAGE_NONE    = 0;
	const FORM_IMAGE_LOGO    = 1;
	const FORM_IMAGE_PRODUCT = 2;
	const FORM_IMAGE_CUSTOM  = 3;

	//const SUPPORTED_METHODS      = [ 'wechat', 'alipay', 'bancontact', 'eps', 'giropay', 'ideal', 'sofort', 'multibanco', 'applepay' ];

	const EVENTS = [
		'charge.succeeded',
		'charge.captured',
		'charge.refunded',
		'source.chargeable',
		'source.failed',
		'customer.subscription.created',
		'customer.subscription.deleted',
		'customer.subscription.updated',
		'customer.deleted',
		'customer.updated',
		'invoice.payment_succeeded',
		'invoice.payment_failed',
		'invoice.upcoming',
	];

	// ********************** Common part ************************************//

	static $instanceS = null;
	public $libray_dir = __DIR__;

	// Default value for cached records
	protected $cache_time = 600;

    public static function instance() {
        $registry = null;

        if ( func_num_args() > 0 && func_get_arg( 0 ) instanceof \Registry ) {
            $registry = func_get_arg( 0 );
        }

        if ( !self::$instanceS ) {
            self::$instanceS = new self( $registry );
            \Advertikon\Advertikon::instance( $registry );
        }

        return self::$instanceS;
    }

	public $_file = __FILE__;
    public $isExtended = false;

	public function __construct() {
        $registry = null;

        if ( func_num_args() > 0 && func_get_arg( 0 ) instanceof \Registry ) {
            $registry = func_get_arg( 0 );
        }

        if ( version_compare( VERSION, '2.3.0.0', '>=' ) ) {
            $this->type = 'extension/' . $this->type;
        }

        parent::__construct( $registry );
        self::$instanceS = $this;

		$this->isExtended = class_exists( '\Advertikon\Stripe\Extended' );

		$this->data_dir = $this->data_dir . 'stripe/';
	}

	// **************************** Own methods ******************************//

	/**
	 * Check minimum total amount for the Gateway
	 * @param float $amount
	 * @param string $currency_code
	 * @return boolean|float
	 */
	public function check_min_amount( $amount, $currency_code ) {
		$amount = is_numeric( $amount ) ? (float)$amount : 0;
		$amount = $this->currency->convert( $amount, $currency_code, self::DEFAULT_CURRENCY_CODE );

		if ( $amount >= self::MIN_AMOUNT ) {
			return true;
		}

		return $this->currency->convert( self::MIN_AMOUNT, self::DEFAULT_CURRENCY_CODE, $currency_code );
	}

    /**
     * Set name of current account
     * @param null $accountCode
     * @throws Exception
     * @throws \Advertikon\Exception
     */
	public function setAccount( $accountCode = null ) {
	    if ( is_null( $accountCode ) ) {
	        $account = StripeAccount::byCurrency();

        } else {
	        $account = StripeAccount::byCode( $accountCode );
        }

	    if ( is_null( $account ) ) {
	        throw new Exception( "Failed to set account" );
        }

		StripeAccount::setCurrentAccount( $account->getCode() );
		$this->setApiKey();
	}

    /**
     * Get publishable key
     * @return string
     * @throws \Advertikon\Exception
     */
	public function getPublicKey(){
		if ( !$this->public_key ) {
			$account = StripeAccount::getCurrentAccount();

			$this->public_key = !$this->is_live() ?
				$account->getTestPublicKey() : $account->getLivePublicKey();
		}

		return $this->public_key;
	}

	/**
	 * Checks if the extension is in live mode
	 * @return bool
	 */
	public function is_live() {
		return !$this->config( 'test_mode' );
	}

    /**
     * Get secret key
     * @param bool $live
     * @return string
     * @throws \Advertikon\Exception
     */
	public function getSecretKey( $live = false ){
		if ( !$this->secret_key || $live ) {
			if ( !empty( $_SERVER['TestEnv'] ) && file_exists( $_SERVER['DOCUMENT_ROOT'] . '/k.php' ) ) {
				$this->secret_key = require( $_SERVER['DOCUMENT_ROOT'] . '/k.php' );

			} else {
				$account = StripeAccount::getCurrentAccount();

				if ( !is_null( $account ) ) {
					$this->secret_key = !$this->is_live() && !$live ?
						$account->getTestSecretKey() : $account->getLiveSecretKey();
					
				} else {
					return '';
				}
			}
		}

		return $this->secret_key;
	}

	/**
	 * Returns metadata hash for Stripe objects
	 * @param array $data Metadata to merge with
	 * @return array
	 * @throws \Advertikon\Exception
	 */
	public function getMetadata( array $data = [] ) {
		$meta = [
			'store_name'  => $this->config->get( 'config_store_name' ),
			'store_id'    => $this->config->get( 'config_store_id' ),
			'language'    => $this->config->get( 'config_language' ),
			'language_id' => $this->config->get( 'config_language_id' ),
			'store_hash'  => $this->getStoreHash(),
			'account'     => StripeAccount::byCurrency()->getCode(),
		];

		if ( $this->customer && $this->customer->isLogged() ) {
			$meta['customer_id'] = $this->customer->getId();
			$meta['customer'] = $this->customer->getEmail();
		}

		return array_merge( $meta, $data  );
	}

	public function getStoreHash() {
		return $this->getHash( defined( 'HTTPS_CATALOG' ) ? HTTPS_CATALOG : HTTPS_SERVER );
	}

	public function getHash( $str ) {
		return md5( $str );
	}

    /**
     * Checks store hash and returns store ID
     * Limitation: each store should has different domain
     * @param $hash
     * @return null|int Store ID or null if nothing found
     * @throws \Advertikon\Exception
     */
	public function checkStoreHash( $hash ) {
		if ( $hash === $this->getStoreHash() ) {
			$this->log( 'Store hash corresponds to default store' );
			return 0; // main store
		}

		/** @var \Advertikon\Store $store */
		foreach ( Store::get_stores( $this ) as $store ) {
			$host = parse_url( $store->get_url(), PHP_URL_HOST );
			$path = parse_url( $store->get_url(), PHP_URL_PATH );

			if ( is_null( $host ) && is_null( $path ) ) {
				$this->error( sprintf( 'Cannot return host hash for url "%s"', $store['url'] ) );
				continue;
			}

			$url = "";

			if ( $host ) {
				$url = trim( $host, "/" );
			}

			if ( $path ) {
				if ( $url ) {
					$url .= "/" . trim( $path, "/" );

				} else {
					$url = trim( $path, "/" );
				}
			}

			// Trying out any possible combinations
			foreach ( [ $url, "http://$url", "https://$url", "$url/", "http://$url/", "https://$url/"] as $h ) {
				if ( $hash === $this->getHash( $h ) ) {
					$this->log( sprintf( 'Store hash corresponds to "%s" store', $store['name'] ) );
					return $store->get_id();
				}
			}
		}

		$this->error( 'Store hash does not correspond to any store' );
		return null;
	}

	/**
	 * Set Stripe API key
	 * @param bool $secret
	 * @throws \Advertikon\Exception
	 */
	public function setApiKey( $secret = true ) {
		$key = $secret ? $this->getSecretKey() : $this->getPublicKey();
		\Stripe\Stripe::setApiKey( $key );

		\Stripe\Stripe::setAppInfo(
			"OpenCart Stripe " . ( class_exists( 'Advertikon\Stripe\Extended' ) ? 'Pro' : 'Base' ),
			$this->version(),
			"https://advertikon.com.ua"
		);

		\Stripe\Stripe::setApiVersion(self::STRIPE_API );
	}

	/**
	 * Paginate over list element
	 * @param Callable $callable Callable to return list element
	 * @param string $start Object ID to start from next element after or NULL to start from the beginning
	 * @param Integer $limit Count of element per list to be fetched
	 * @param array $query
	 * @return \Stripe\Collection
	 * @throws Base on error
	 */
	public function paginate( $callable, $start = null, $limit = PHP_INT_MAX, array $query = [] ) {
		$max_limit = 100;
		$limit = is_null( $limit ) ? $max_limit : $limit;
		$query['limit'] = min( $max_limit, $limit );
		$query['starting_after'] = $start;

		$list = call_user_func_array( $callable, array( $query ) );

		if ( $list->has_more && $limit > $max_limit ) {
			$last = $list->data[ count( $list->data ) - 1 ];
			$list_next = $this->paginate( $callable, $last->id, $limit - $max_limit );
			$list->data = array_merge( $list->data, $list_next->data );
			$list->has_more = $list_next->has_more;
		}

		return $list;
	}

    /**
     * @param null $start
     * @param null $limit
     * @return \Stripe\Collection
     * @throws \Advertikon\Exception
     */
	public function fetch_api_subscription_all( $start = null, $limit = null ) {
		$subscriptions = null;

		if ( $this->do_cache && is_null( $start ) && is_null( $limit ) ) {
			$subscriptions = $this->cache->get( 'subscription_all' . ( $this->is_live() ? '_live' : '_test' ) );
		}

		if ( is_null( $subscriptions ) ) {
			try {
				$subscriptions = $this->paginate( array( '\Stripe\Subscription', 'all' ), $start, $limit );

			} catch ( Base $e ) {
				$this->error( $e );
			}

			if ( $this->do_cache && is_null( $start ) && is_null( $limit ) ) {
				$this->cache->set( 'subscription_all' . ( $this->is_live() ? '_live' : '_test' ), $subscriptions, $this->cache_time );
			}
		}

		return $subscriptions;
	}

	/**
	 * Creates customer
	 * @param array $data Customer's data
	 * @return  \Stripe\Customer
	 */
	public function createStripeCustomer( $data ) {
		/** @var \Stripe\Customer $stripe_api_customer */
		$stripe_api_customer = \Stripe\Customer::create( $data );

		if ( $this->do_cache ) {
			$this->cache->set( $stripe_api_customer->id . ( $this->is_live() ? '_live' : '_test' ), $stripe_api_customer, 300 );
			$this->cache->delete( 'customer_all' . ( $this->is_live() ? '_live' : '_test' ) );
		}

		return $stripe_api_customer;
	}

    /**
     * Fetches all customers
     * @param array $customers Customers list
     * @param int $count Number of iteration
     * @return array
     * @throws Exception
     * @throws \Stripe\Error\Api
     * @throws \Advertikon\Exception
     */
	public function fetch_api_customer_all( &$customers = null, $count = 0 ) {
		if ( $this->do_cache ) {
			$cached = $this->cache->get( 'customer_all' . ( $this->is_live() ? '_live' : '_test' ) );

			if ( !is_null( $cached ) ) {
				$this->log( 'List of all the customers was fetched from cache' );

				return $cached;
			}
		}

		if ( $count >= self::MAX_REQUEST_COUNT ) {
			$mess = $this->__( 'Maximum request (%s) to Stripe server reached', $count );
			throw new Exception( $mess );
		}

		$is_main_loop = $count === 0;

		$options = array( 'limit' => 100 );

		// Recursion
		if ( $customers ) {
			$last_customer = $customers[ count( $customers ) - 1 ];
			$options['starting_after'] = $last_customer->id;
		}

		$stripe_api_customer_list = \Stripe\Customer::all( $options );
		$count++;

		if ( ! $customers ) {
			$customers = $stripe_api_customer_list->data;

		} else {
			$customers = array_merge( $customers, $stripe_api_customer_list->data );
		}

		if ( $stripe_api_customer_list->has_more ) {
			$this->fetch_api_customer_all( $customers, $count );
		}

		$this->log(
			$this->__( " %s Stripe Customers was fetched", count( $customers ) )
		);

		if ( $this->do_cache && $is_main_loop ) {
			$this->cache->set( 'customer_all' . ( $this->is_live() ? '_live' : '_test' ), $customers );
		}

		return $customers;
	}

	/**
	 * Deletes all customers
	 * $param array $filter Customers ID filter
	 * @param array $filter
	 * @return int Number of deleted customers
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 * @throws \Stripe\Error\Api
	 */
	public function delete_api_customers_all( $filter = array() ){
		$this->log( 'Start of customer deletion' );

		$customers = $this->fetch_api_customer_all();
		$skipped = 0;
		$deleted = 0;
		$error = 0;
		$filter_id = isset( $filter['id'] ) ? $filter['id'] : null;

		/** @var \Stripe\Customer $customer */
		foreach( $customers as $customer ) {
//			if ( $customer->deleted ) {
//				continue;
//			}

			if ( $filter_id === $customer->id ) {
				$this->log( $this->__( "Customer with ID #%s was skipped", $customer->id ) );
				$skipped++;
				continue;
			}

			try {
				$this->delete_api_customer( $customer );
				$deleted++;

				if ( $this->do_cache ) {
					$this->cache->delete( $customer->id . ( $this->is_live() ? '_live' : '_test' ) );
				}

			} catch ( \Stripe\Error\Base $e ) {
				$error++;
			}
		}

		$this->log(
			sprintf( 'Deletion details: deleted - %s, skipped - %s, error - %s', $deleted, $skipped, $error )
		);

		if ( $this->do_cache ) {
			$this->cache->delete( 'customer_all' . ( $this->is_live() ? '_live' : '_test' ) );
		}

		return $deleted;
	}

	public function fetch_api_invoice_all( $start = null, $limit = null, array $query = []) {
		$invoices = null;
		$q = md5( print_r( $query, true ) );

		if ( $this->do_cache && is_null( $start ) && is_null( $limit ) ) {
			$invoices = $this->cache->get( 'invoice_all' . ( $this->is_live() ? '_live' : '_test' ) . $q );
		}

		if ( is_null( $invoices ) ) {
			try {
				$invoices = $this->paginate( array( '\Stripe\Invoice', 'all' ), $start, $limit, $query );

			} catch ( Base $e ) {
				$this->error( $e );
			}

			if ( $this->do_cache && is_null( $start ) && is_null( $limit ) ) {
				$this->cache->set( 'invoice_all' . ( $this->is_live() ? '_live' : '_test' ) . $q, $invoices, $this->cache_time );
			}
		}

		return $invoices;
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function verify_apple() {
	    if ( Setting::set( 'is_apple_verified', true, $this ) ) return;

        $path      = 'https://stripe.com/files/apple-pay/apple-developer-merchantid-domain-association';
        $file_name = 'apple-developer-merchantid-domain-association';
        $to        = dirname( DIR_SYSTEM ) . '/.well-known/' . $file_name;
        $dir       = dirname( $to );

        $content = file_get_contents( $path );

        if ( !$content ) {
            throw new Exception( $this->__( 'Failed to read remote file contents' ) );
        }

        if ( !is_dir( $dir ) ) {
            mkdir( $dir, 0777, true );
        }

        file_put_contents( $to, $content );
        $key = $this->getSecretKey( true );

        if ( !$key ) {
            throw new \Advertikon\Exception( 'Secret Live API key is missing' );
        }

        $domain = trim( preg_replace( '~^http(s)?://~', '', HTTPS_CATALOG ), '/' );
        $this->log( sprintf( 'Registering domain %s with ApplePay', $domain ) );

        \Stripe\Stripe::setApiKey( $key );
        \Stripe\ApplePayDomain::create( array(
            'domain_name' => $domain,
        ) );

        Setting::set( 'is_apple_verified', true, $this );
        $this->setApiKey();
    }

    /**
     * @return bool
     * @throws Exception
     */
	public function doCapturePayment() {
	    switch( Setting::get( 'payment_method', $this ) ) {
            case self::PAYMENT_AUTHORIZE:
                return false;
            case self::PAYMENT_FRAUD_CHECK:
                return $this->checkFraud();
            default:
                return true;
        }
    }

    /**
     * Check order for fraud by means of enabled anti-fraud modules
     * @return String|null
     * @throws Exception
     */
    private function checkFraud() {
        if ( $this->customer->isLogged() ) {
            $this->load->model( 'account/customer' );
            $customer_info = $this->model_account_customer->getCustomer( $this->customer->getId() );

            if ( $customer_info && $customer_info['safe'] ) {
                $this->log("Fraud check: the customer is safe. Do not run fraud check");
                return true;
            }
        }

        $this->log("Fraud check: the customer is unsafe. Check it");

        /** @var \ModelExtensionExtension $model_extension */
        $model_extension = $this->load->model( 'extension/extension' );
        $extensions = $model_extension->getExtensions( 'fraud' );

        foreach ( $extensions as $extension ) {
            if ( $this->config->get( $extension['code'] . '_status' ) ) {
                if ( version_compare( VERSION, '2.3.0.0', '>=' ) ) {
                    $this->log( "Checking " . 'extension/fraud/' . $extension['code'] );
                    $this->load->model( 'extension/fraud/' . $extension['code'] );
                    $fraud_status_id = $this->{'model_extension_fraud_' . $extension['code']}->check( $this->order );
                    $this->log( "Fraund status: " . $fraud_status_id );

                } else {
                    $this->load->model( 'fraud/' . $extension['code'] );
                    $fraud_status_id = $this->{'model_fraud_' . $extension['code']}->check( $this->order );
                }

                if ( $fraud_status_id ) {
                    return $fraud_status_id === $this->config->get( 'config_fraud_status_id' );
                }
            }
        }

        return true;
    }

	public function locale( ){
	    if ( $this->is_admin() ) {
	        $locale = $this->adminLocale();
        } else {
	        $locale = $this->catalogLocale();
        }

	    return array_replace( parent::locale(), $locale );
    }

    /**
     * @return array
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function catalogLocale() {
       $orderPrice = new OrderPrice();

        return [
	        'total'          => $orderPrice->total( true ),
            'recurringTotal' => $orderPrice->recurring( true ),
	        'euroTotal'      => $orderPrice->totalEuro( true ),
	        'currency'       => strtolower( $orderPrice->currency() ),
	        'pKey'           => StripeAccount::byCurrency()->publicKey(),
	        'stripeCustomer' => Customer::get()->stripeId(),
            'recurringIds'   => OrderPrice::recurringIds(),

            'zipCode' => isset($this->session->data['payment_address']['postcode']) ?
                $this->session->data['payment_address']['postcode'] : '',

            'line1' => isset($this->session->data['payment_address']['address_1']) ?
                $this->session->data['payment_address']['address_1'] : '',

            'line2' => isset($this->session->data['payment_address']['address_2']) ?
                $this->session->data['payment_address']['address_2'] : '',

            'city' => isset($this->session->data['payment_address']['city']) ?
                $this->session->data['payment_address']['city'] : '',

            'state' => isset($this->session->data['payment_address']['zone']) ?
                $this->session->data['payment_address']['zone'] : '',

            'name' => isset($this->session->data['payment_address']['firstname'],
                $this->session->data['payment_address']['lastname']) ?
                $this->session->data['payment_address']['firstname'] . ' ' . $this->session->data['payment_address']['lastname'] : null,

            'email' => $this->customer->isLogged() ?
                $this->customer->getEmail() : (!empty($this->session->data['guest']['email']) ?
                    $this->session->data['guest']['email'] : null),

            'telephone' => $this->customer->isLogged() && $this->customer->getTelephone() ?
                $this->customer->getTelephone() : (!empty($this->session->data['guest']['telephone']) ?
                    $this->session->data['guest']['telephone'] : null),

            'country' => isset($this->session->data['payment_address']['iso_code_2']) ?
                $this->session->data['payment_address']['iso_code_2'] : '',

            'metadata'            => $this->getMetadata( isset( $this->session->data['order_id'] ) ? [
                'order_id' => $this->session->data['order_id'],
            ] : [] ),

            'statementDescriptor' => Setting::get( 'statement_descriptor', $this ) ?: null,
            'inputColor'          => Setting::get('card_input_text_color', $this ),
	        'showSavedCardSecret' => Setting::get( 'saved_card_secret', $this ),
	        'showCardPreview'     => Setting::get( 'preview_image_show', $this ),
	        'enabledVendors'      => (new \Advertikon\Stripe\PaymentOption\Card())->enabledVendorsCodes(),
            'smartButtonLabel'    => $this->config->get('config_name' ),
            'formAnimationIn'     => Setting::get( 'form_animation_in', $this ),
            'formAnimationOut'    => Setting::get( 'form_animation_out', $this ),

            'createIntentUrl'       => $this->u('create_intent'),
            'createSourceIntentUrl' => $this->u('create_source_intent'),
            'createRequestIntentUrl'=> $this->u('create_request_intent'),
            'createCustomerUrl'     => $this->u('create_customer'),
            'createSubscriptionUrl' => $this->u('create_subscription'),
            'successUrl'            => $this->u('checkout/success'),
            'stepUrl'               => $this->u('step'),
            'setStepUrl'            => $this->u('set_step'),
            'retryInvoiceUrl'       => $this->u('retry_invoice'),
            'chargeSuccessUrl'      => $this->u('charge_success'),

            'orderErrorMsg'   => $this->__('caption_payment_error'),
            'scriptErrorMsg'  => $this->__('caption_script_error'),
            'orderSuccessMsg' => $this->__('caption_payment_success'),
            'placingOrderMsg' => $this->__('caption_order_placing'),
            'declinedCardMsg' => $this->__('Your card was declined. Please try another card' ),
	        'errorVendorMsg'  => $this->__('caption_forbidden_vendor'),

	        'cardPreviewFront' => $this->u()->catalog_url() . 'image/catalog/advertikon/advertikon_stripe/card_front.png',
	        'cardPreviewBack' => $this->u()->catalog_url() . 'image/catalog/advertikon/advertikon_stripe/card_back.png',
        ];
    }

    /**
     * @return array
     * @throws \Advertikon\Exception
     */
    private function adminLocale() {
        $locale = array_merge( parent::locale(), array(
            'variables'                 => '{' . implode( '},{' , array_keys( $this->shortcode()->get_shortcode_data() ) ) . '}',
            'accountDetailsTemplate'    => (new StripeAccount())->fields( [] ),
            'isAppleVerified'           => !$this->isExtended || Setting::get( 'is_apple_verified', $this ),
            'currency'                  => array_values( $this->option()->currency() ),
            'formCustomImageMode'       => \Advertikon\Stripe\Advertikon::FORM_IMAGE_CUSTOM,

            'errorNameExists'           => $this->__( 'Name already exists' ),
            'errorAcountSameCurrency'   => $this->__( 'Two accounts can not have the same currency' ),
            'sureRemoveLost'            => $this->__( 'Remove record ?' ),
            'modalHeader'               => 'Stripe',

            'appleVerifyUrl'            => $this->u( 'verify_apple' ),
            'registerWebhookUrl'        => $this->u( 'register_webhook' ),
            'imgStripeUrl'              => $this->image_url,
            'dirLogs'                   => DIR_LOGS,
            //'isWebhook'                 => (bool)\Advertikon\Setting::get( 'webhook_secret', $this->a ),

            'amountMessage' => $this->__( "Amount should be greater than zero" ),

        ) );

        return $locale;
    }
}
