<?php
/**
 * Advertikon button.php Class
 * @author Advertikon
 * @package
 * @version 5.0.44  
 */

namespace Advertikon\Stripe;


use Advertikon\Element\Slider;
use Advertikon\Setting;
use Advertikon\Sql;

class Button {
	/** @var Advertikon */
	private $a;

	private $error = [];
	private $cartData = [];
	private $oldCart = [];
	private $data = [];

	/**
	 * Button constructor.
	 * @param array $data
	 * @throws \Advertikon\Exception
	 */
	public function __construct( array $data = [] ) {
		$this->a = Advertikon::instance();
		$this->data = $data;
	}

	public function setProductId( $product_id ) {
		$this->data['product_id'] = $product_id;
	}

	/**
	 * @param $product_id
	 * @return bool|string
	 * @throws \Advertikon\Exception
	 */
	public function doShow() {
		if( !Setting::get( 'button', $this->a ) || !Setting::get( 'status', $this->a ) ) {
			return false;
		}

        if( !$this->a->customer->isLogged() ) {
            $this->a->log->write( 'Stripe: Payment button is disabled - customer is not logged in' );
            return false;
        }

		$product = $this->a->get_product( $this->data['product_id'] );

		// Missing stock
		if( $product['quantity'] < $product['minimum']  && !$this->a->config->get( 'config_stock_checkout' ) ) {
			$this->a->log->write( 'Stripe: Payment button is disabled - low stock' );
			return false;
		}

		// If product is download-able and customer is not logged - do not show button
//		if( $product['download'] ) {
//			$this->a->log->write( 'Stripe: Payment button is disabled - product is downloadable and customer is not logged in' );
//			return false;
//		}

		// If product requires shipping but customer is not logged or there is no default shipping method - skip
		if(	$product['shipping'] && !Setting::get( 'button_shipping', $this->a ) ) {
			$this->a->log->write( 'Stripe: Payment button is disabled - product requires shipping but default shipping method is not defined' );
			return false;
		}

		return true;
	}

    /**
     * @throws Exception
     * @throws \Advertikon\Exception
     */
	public function prepare() {
        $this->getCartData();
        $this->setCart();
		$this->setShipping();
		$this->setPayment();
	}

	/**
	 * @return array
	 * @throws \Advertikon\Exception
	 * @throws Exception
	 */
	public function getPrice() {
		$this->prepare();
		$orderPrice = new OrderPrice();

        $currency       = $orderPrice->currency();
        $total          = $orderPrice->total() + $orderPrice->recurring();
        $totalFormatted = $this->a->currency->format( $total, $currency, 1, true );
        $euroTotal      = $orderPrice->totalEuro() + $orderPrice->recurringEuro();
        $euroTotalCents = $orderPrice->totalEuro( true ) + $orderPrice->recurringEuro( true );
        $centsTotal     = $orderPrice->total( true ) + $orderPrice->recurring( true );
        $totals         = $orderPrice->totals();
        $recurringPrice = $orderPrice->recurringStore();

        $buttonText = str_replace( '{{amount}}', $totalFormatted, $this->a->__( 'caption_button_caption' ) );

        $this->resetCart();

		$description =
			'<table style="width: 100%">';

		if ( $recurringPrice ) {
			$totals[] = [
				'code'       => 'recurring',
				'title'      => $this->a->__( 'Subscription' ),
				'value'      => $recurringPrice,
				'sort_order' => 5,
			];
		}

		usort( $totals, function( array $a, array $b ) { return (int)$a['sort_order'] - (int)$b['sort_order']; } );

		foreach( $totals as $line ) {
			if ( $line['code'] === 'total' ) {
				$line['value'] += $recurringPrice;
				$total_initial = $line['value'];
			}

			$t = $orderPrice->convert( $line['value'] );

			$description .=
				'<tr>' .
				'<td>' . $line['title'] . '</td>' .
				'<td>' . $this->a->currency->format( $t, $orderPrice->currency(), 1, true ) . '</td>' .
				'</tr>';
		}

		$description .=
			'</table>';

		$ret['description']         = $description;
		$ret['totals']              = $totals;
		$ret['total']               = $total;
		$ret['total_formatted']     = $totalFormatted;
		$ret['total_cents']         = $centsTotal;
		$ret['total_euro']          = $euroTotal;
		$ret['total_cents_euro']    = $euroTotalCents;
		$ret['button_text']         = $buttonText;
		$ret['currency']            = $currency;

		return $ret;
	}

    /**
     * @throws Exception
     * @throws \Advertikon\Exception
     */
	public function addOrder() {
	    $ret = [];

	    try {
		    $this->getPrice();
		    $result = false;
		    $this->prepare();

		    if ( isset( $this->a->session->data['order_id'] ) ) {
			    unset( $this->a->session->data['order_id'] );
		    }

		    $this->a->load->controller('checkout/confirm' );

		    if ( isset( $this->a->session->data['order_id'] ) ) {
			    $result = true;
		    }

		    if ( $result ) {
			    $ret['success'] = 'ok';

		    } else {
			    $ret['errorMsg'] = $this->a->__( 'Unable to place the order' );
		    }

	    } catch ( \Exception $e ) {
		    if ( $this->hasError() ) {
			    $ret['error'] = $this->error;
			    return $ret;

		    } else {
		    	$ret['error'] = $this->a->__( 'Error' );
		    }
	    }


        return $ret;
    }

	/**
	 * @throws \Advertikon\Exception
	 */
	private function getCartData() {
		$this->a->load->language('checkout/cart');
		$data = $this->data;

		if (isset($data['product_id'])) {
			$product_id = (int)$data['product_id'];
		} else {
			$product_id = 0;
		}

		$this->a->load->model('catalog/product');
		$product_info = $this->a->model_catalog_product->getProduct($product_id);

		if ($product_info) {
			if (isset($data['quantity']) && ((int)$data['quantity'] >= $product_info['minimum'])) {
				$quantity = (int)$data['quantity'];
			} else {
				$quantity = $product_info['minimum'] ? $product_info['minimum'] : 1;
			}

			if (isset($data['option'])) {
				$option = array_filter($data['option']);
			} else {
				$option = array();
			}

			$product_options = $this->a->model_catalog_product->getProductOptions( $product_id );

			foreach ($product_options as $product_option) {
				if ($product_option['required'] && empty($option[$product_option['product_option_id']])) {
					$this->error['option'][$product_option['product_option_id']] = sprintf($this->a->language->get('error_required'), $product_option['name']);
				}
			}

			if (isset($data['recurring_id'])) {
				$recurring_id = $data['recurring_id'];
			} else {
				$recurring_id = 0;
			}

			$recurrings = $this->a->model_catalog_product->getProfiles($product_info['product_id']);

			if ($recurrings) {
				$recurring_ids = array();

				foreach ($recurrings as $recurring) {
					$recurring_ids[] = $recurring['recurring_id'];
				}

				if (!in_array($recurring_id, $recurring_ids)) {
					$this->error['recurring'] = $this->a->language->get('error_recurring_required');
				}
			}

			if ( $this->hasError() ) {
				throw new \Advertikon\Exception( 'Error' );
			}

			$this->cartData = [ 'product_id' => $product_id, 'quantity' => $quantity, 'option' => $option, 'recurring_id' => $recurring_id, ];
		}
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	private function setCart() {
		if ( version_compare( VERSION, '2.3', '>=' ) ) {
			$this->setCart23();

		} else if ( version_compare( VERSION, '2.1', '>=' ) ) {
			$this->setCart22();

		} else {
			$this->setCart20();
		}
	}

	/**
	 * @throws Exception
	 */
	private function setCart20() {
		$this->a->session->data['adk_old_cart'] = isset( $this->a->session->data['cart'] ) ? $this->a->session->data['cart'] : [];

		$product = [
			'product_id'   => (int)$this->cartData['product_id'],
			'option'       => isset( $this->cartData['option'] ) ? $this->cartData['option'] : [],
			'recurring_id' => isset( $this->cartData['recurring_id'] ) ? (int)$this->cartData['recurring_id'] : 0,
		];

		$key = base64_encode( serialize( $product ) );
		$qty = (int)$this->cartData['quantity'];

		if ( $qty > 0 ) {
			$this->a->session->data['cart'] = [];
			$this->a->session->data['cart'][$key] = $qty;

		} else {
			throw new Exception( 'Zero quantity' );
		}
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	private function setCart23() {
		$apiId = ( isset( $this->a->session->data['api_id'] ) ? (int)$this->a->session->data['api_id'] : 0 );

        $this->a->session->data['adk_old_cart'] = Sql::select( 'cart', true )
			->where('api_id')->equal($apiId)
			->where('customer_id')->equal($this->a->customer->getId())
			->where('session_id')->equal($this->a->session->getId())->end()->run();

		Sql::delete( 'cart', true )
			->where('api_id')->equal($apiId)
			->where('customer_id')->equal($this->a->customer->getId())
			->where('session_id')->equal($this->a->session->getId())->end()->run();

		$count = Sql::insert( 'cart', true )
			->set([
				"api_id"       => $apiId,
				"customer_id"  => $this->a->customer->getId(),
				"session_id"   => $this->a->session->getId(),
				"product_id"   => $this->cartData['product_id'],
				"recurring_id" => isset( $this->cartData['recurring_id'] ) ? $this->cartData['recurring_id'] : 0,
				"option"       => isset( $this->cartData['option'] ) ? json_encode( $this->cartData['option'] ) : '[]',
				"quantity"     => $this->cartData['quantity'],
				"date_added"   => new Sql\RawValue( 'NOW()'),
			])->run();

		if ( $count === 0 ) {
			$this->resetCart();
			throw new Exception( "Failed to update cart" );
		}
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	private function setCart22() {
        $this->a->session->data['adk_old_cart'] = Sql::select( 'cart', true )
			->where('customer_id')->equal($this->a->customer->getId())
			->where('session_id')->equal($this->a->session->getId())->end()->run();

		Sql::delete( 'cart', true )
			->where('customer_id')->equal($this->a->customer->getId())
			->where('session_id')->equal($this->a->session->getId())->end()->run();

		$count = Sql::insert( 'cart', true )
			->set([
				"customer_id"  => $this->a->customer->getId(),
				"session_id"   => $this->a->session->getId(),
				"product_id"   => $this->cartData['product_id'],
				"recurring_id" => isset( $this->cartData['recurring_id'] ) ? $this->cartData['recurring_id'] : 0,
				"option"       => isset( $this->cartData['option'] ) ? json_encode( $this->cartData['option'] ) : '[]',
				"quantity"     => $this->cartData['quantity'],
				"date_added"   => new Sql\RawValue( 'NOW()'),
			])->run();

		if ( $count === 0 ) {
			$this->resetCart();
			throw new Exception( "Failed to update cart" );
		}
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	public function resetCart() {
		if ( version_compare( VERSION, '2.3', '>=' ) ) {
			$this->resetCart23();

		} else if ( version_compare( VERSION, '2.2.0.0', '=' ) ) {
			$this->resetCart22();

		} else {
			$this->resetCart20();
		}

		if( isset( $this->a->session->data['order_id'] ) ) {
		    unset( $this->a->session->data['order_id'] );
        }
	}

	private function resetCart20() {
		$this->a->session->data['cart'] = $this->a->session->data['adk_old_cart'];
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	private function resetCart23() {
		$apiId = ( isset( $this->a->session->data['api_id'] ) ? (int)$this->a->session->data['api_id'] : 0 );

		Sql::delete( 'cart', true )
			->where('api_id')->equal($apiId)
			->where('customer_id')->equal($this->a->customer->getId())
			->where('session_id')->equal($this->a->session->getId())->end()->run();

		foreach( $this->a->session->data['adk_old_cart'] as $line ) {
			Sql::insert( 'cart', true )->set( $line )->run();
		}
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	private function resetCart22() {
		Sql::delete( 'cart', true )
			->where('customer_id')->equal($this->a->customer->getId())
			->where('session_id')->equal($this->a->session->getId())->end()->run();

		foreach( $this->a->session->data['adk_old_cart'] as $line ) {
			Sql::insert( 'cart', true )->set( $line )->run();
		}
	}

	private function readError( $default = '' ) {
		$out = $this->a->response->getOutput();

		if ( $out ) {
			$response = @json_decode( $out, true );

			if ( $response ) {
				$error = isset( $response['error']['warning'] ) ? $response['error']['warning'] : $default;
				$this->a->error( $error );
			}

			return true;
		}

		return false;
	}

	private function hasError() {
		return count( $this->error ) > 0;
	}

	private function setShipping() {
		if ( !$this->a->cart->hasShipping() ) {
			return;
		}

		$shipping_code = Setting::get( 'button_shipping', $this->a );

		$this->a->request->post['shipping_address'] = 'existing';
		$this->a->request->post['address_id'] = $this->a->customer->getAddressId();
		$this->a->load->controller( 'checkout/shipping_address/save' );

		$this->readError( 'Failed to save shipping address' );

		$this->a->load->controller( 'checkout/shipping_method' );
		$this->a->request->post['shipping_method'] = $shipping_code;
		$this->a->request->post['comment'] = '';

		// Take first sub-method
		if ( isset( $this->a->session->data['shipping_methods'][ $shipping_code ]['quote'] ) ) {
			$c = reset($this->a->session->data['shipping_methods'][ $shipping_code ]['quote'] )[ 'code' ];
			$this->a->request->post['shipping_method'] =  $c;
		}

		$this->a->load->controller( 'checkout/shipping_method/save' );
		$this->readError( 'Failed to save shipping method' );
	}

	private function setPayment() {
		$payment_code = $this->a->code;
		$this->a->request->post['payment_address'] = 'existing';
		$this->a->request->post['address_id'] = $this->a->customer->getAddressId();
		$this->a->load->controller( 'checkout/payment_address/save' );

		$this->readError( 'Failed to save payment address' );

		$this->a->load->controller( 'checkout/payment_method' );
		$this->a->request->post['comment'] = '';
		$this->a->request->post['agree'] = '0';
		$this->a->request->post['payment_method'] = $payment_code;
		$this->a->load->controller( 'checkout/payment_method/save' );
		$this->readError( 'Failed to save payment method' );
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
	public function get() {
		$orderPrice = new OrderPrice();
		$paymentForm = new PaymentForm( true );

        $currency       = '';
        $total          = 0;
        $totalFormatted = '';
        $euroTotal      = 0;
        $centsTotal     = '';

        try {
            $this->prepare();

            $currency       = $orderPrice->currency();
            $total          = $orderPrice->total() + $orderPrice->recurring();
            $totalFormatted = $this->a->currency->format( $total, $currency, 1, true );
            $euroTotal      = $orderPrice->totalEuro() + $orderPrice->recurringEuro();
            $centsTotal     = $orderPrice->total( true ) + $orderPrice->recurring( true );

            $this->resetCart();
        } catch ( \Advertikon\Exception $e ) {}

        $buttonText     = str_replace( '{{amount}}', $totalFormatted, $this->a->__( 'caption_button_caption' ) );

		$data = $paymentForm->getData();

		$locale = [
		    'priceUrl'       => $this->a->u()->url('button_get_price'),
		    'createOrderUrl' => $this->a->u()->url('create_order'),
            'isPayButton'    => true,
        ];

		$data['requireJs']       = $this->a->requireJs( ['stripe/pay_button' => 'main' ], [], $locale );
		$data['is_extended']     = $this->a->isExtended;
		$data['total']           = $total;
		$data['currency']        = $currency;
		$data['total_formatted'] = $totalFormatted;
		$data['popup']           = $paymentForm->isPopUp();

		$textHeight        = Setting::get( 'button_text_height', $this->a, Slider::def(null,null,20) );
		$borderRadius      = Setting::get( 'button_radius', $this->a, Slider::def(null,null,5) );
		$marginVertical    = Setting::get( 'button_margin_vertical', $this->a, Slider::def(null,null,0) );
		$marginHorizontal  = Setting::get( 'button_margin_horizontal', $this->a, Slider::def(null,null,0) );
		$paddingVertical   = Setting::get( 'button_padding_v', $this->a, Slider::def(null,null,5) );
		$paddingHorizontal = Setting::get( 'button_padding_h', $this->a, Slider::def(null,null,0) );
		$backgroundColor   = Setting::get( 'button_color', $this->a, '#008cdd' );
		$color             = Setting::get( 'button_text_color', $this->a, '#ffffff' );

		$buttonElement = new \Advertikon\Element\Bootstrap\Button( $buttonText );
		$buttonElement
			->id('adk-stripe-button' )
			->getClass()->add( 'adk-one-click adk-button' )
			->style()
				->color( $color )
				->background( $backgroundColor )
				->fontSize( Slider::getMax( $textHeight ) )
				->borderRadius( Slider::getMax( $borderRadius ) )->stop()
				->margin()
					->top( Slider::getMax( $marginVertical ) )
					->bottom( Slider::getMax( $marginVertical ) )
					->left( Slider::getMax( $marginHorizontal ) )
					->right( Slider::getMax( $marginHorizontal ) )
				->stop()
				->padding()
					->top( Slider::getMax( $paddingVertical ) )
					->bottom( Slider::getMax( $paddingVertical ) )
					->left( Slider::getMax( $paddingHorizontal ) )
					->right( Slider::getMax( $paddingHorizontal ) )
				->stop();

		if( Setting::get( 'button_full_width', $this->a ) ) {
			$buttonElement->style()->width('100%');
		}

		$data['button'] = $buttonElement;

		return $data;
	}
}