<?php
/**
 * Advertikon Stripe Order Class
 * @author Advertikon
 * @package Stripe
*@version5.0.44
6
 */

namespace Advertikon\Stripe;


use Advertikon\Setting;
use Advertikon\Sql;
use Stripe\Error\Api;

class Customer {
	const TABLE = 'advertikon_stripe_customer';

	protected $data = [];
	protected $a;

	private $id;
	private $ocId;
	private $stripeId;
	private $isTest;
	private $dateAdded;
	private $account;

    /**
     * @param $customer
     * @param $paymentMethod
     */
	static public function bindPaymentMethod( $customer, $paymentMethod ) {
        $payment_method = \Stripe\PaymentMethod::retrieve( $paymentMethod );
        $payment_method->attach( [ 'customer' => $customer ] );

        // Set the default payment method on the customer
        \Stripe\Customer::update( $customer, [
            'invoice_settings' => [
                'default_payment_method' => $payment_method
            ]
        ]);
    }

	/**
	 * @param null $ocId
	 * @return Customer|null
	 * @throws \Advertikon\Exception
	 */
	public static function get( $ocId = null ) {
		$a = Advertikon::instance();

		if ( is_null( $ocId ) && $a->customer && $a->customer->isLogged() ) {
			$ocId = $a->customer->getId();
		}

		if ( is_null( $ocId ) ) {
			return new self();
		}

		$q = Sql::select( self::TABLE, true )
			->where( 'oc_customer_id' )->equal( $ocId )
			->where( 'is_test' )->equal( self::getMode() )
            ->end()->run();

		return new self( count( $q ) ? $q[0] : null );
	}

	/**
	 * @param $id
	 * @return Customer|null
	 * @throws \Advertikon\Exception
	 */
	public static function getByStripeId( $id ) {
		$q = Sql::select( self::TABLE, true )
			->where( 'stripe_id' )->equal( $id )
			->where( 'is_test' )->equal( self::getMode() )
			->end()->run();

		return new self( count( $q ) ? $q[0] : null );
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	public static function createNew() {
		$a = Advertikon::instance();

		if ( !$a->customer || !$a->customer->isLogged() ) {
			throw new \Advertikon\Exception( 'Customer need to be logged in in order to be saved' );
		}

		$customer = \Advertikon\Customer::get( $a->customer->getEmail() ,$a );
		$addresses = $customer->get_address();

		$data = [
			'email'       => $customer->get_email(),
			'description' => $a->shortcode()->do_shortcode( Setting::get( 'customer_description', $a, '' ) ),
			'metadata'    => $a->getMetadata(),
			'name'        => "{$customer->get_firstname()} {$customer->get_lastname()}",
			'phone'       => $customer->get_telephone(),
		];

		if ( !count( $addresses ) ) {
			$address = $addresses[0];

			$data['address'] = [
				'line1'       => $address->get_address_1(),
				'line2'       => $address->get_address_2(),
				'city'        => $address->get_city(),
				'postal_code' => $address->get_postcode(),
				'state'       => $address->get_zone()
			];
		}

		return $a->createStripeCustomer( $data );
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public static function createNewGuest() {
		$a = Advertikon::instance();

		if ( !isset( $a->session->data['guest'] ) ) {
			throw new Exception( 'No guest found' );
		}

		$customer = $a->session->data['guest'];

		$data = [
			'email'       => $customer['email'],
			'description' => $a->shortcode()->do_shortcode( Setting::get( 'customer_description', $a, '' ) ),
			'metadata'    => $a->getMetadata(),
			'name'        => "{$customer['firstname']} {$customer['lastname']}",
			'phone'       => $customer['telephone'],
		];

		if ( isset( $a->session->data['payment_address'] ) ) {
			$address = $a->session->data['payment_address'];

			$data['address'] = [
				'line1'       => $address['address_1'],
				'line2'       => $address['address_2'],
				'city'        => $address['city'],
				'postal_code' => $address['postcode'],
				'state'       => $address['zone']
			];
		}

		return $a->createStripeCustomer( $data );
	}

	/**
	 * @param array $customerData
	 * @return object
	 * @throws \Advertikon\Exception
	 */
	public static function createNewEmpty( array $customerData = [] ) {
		$a = Advertikon::instance();

		$data = [
			'description' => $a->shortcode()->do_shortcode( Setting::get( 'customer_description', $a, '' ) ),
			'metadata'    => $a->getMetadata(),
		];

		$data = array_merge_recursive( $data, $customerData );

		return $a->createStripeCustomer( $data );
	}

	/**
	 * @param \Stripe\Customer $customer
	 * @return Customer
	 * @throws \Advertikon\Exception
	 */
	static public function createNewWithStripeCustomer ( \Stripe\Customer $customer ) {
		$a = Advertikon::instance();

		return (new self( [
			'oc_customer_id' => $a->customer->getId(),
			'stripe_id'      => $customer->id,
		] ) )
			->save();
	}

	/**
	 * @return string
	 * @throws \Advertikon\Exception
	 */
	static private function getMode() {
		return Setting::get( 'test_mode', Advertikon::instance(), '0' ) ? '1' : '0';
	}

	/**
	 * Customer constructor.
	 * @param array $data
	 * @throws \Advertikon\Exception
	 */
	public function __construct( $data = [] ) {
		$this->a = Advertikon::instance();

		if ( $data ) {
			$this->setData( $data );
		}
	}

	public function ocId() {
		return $this->ocId;
	}

	/**
	 * @return string
	 */
	public function stripeId() {
		return $this->stripeId;
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	public function save() {
		if( !$this->canBeSaved() ) throw new \Advertikon\Exception( 'Customer may not be saved' );

		$this->saveNew();

		return $this;
	}

	public function exists() {
		return !is_null( $this->id );
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	public function delete() {
		Sql::delete(self::TABLE, true )->where('id')->equal($this->id)->end()->run();
	}

	/**
	 * @return \Stripe\Customer
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function fetch() {
		$this->setAccount();
		return \Stripe\Customer::retrieve( $this->stripeId );
	}

	/**
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	private function setAccount() {
		$account = $this->account ?: StripeAccount::getFallbackCode();

		if ( is_null($account ) ) {
			throw new Exception("Undefined account" );
		}

		$this->a->setAccount( $account );
	}

	/**
	 * Fetches all the cards of specific customer
	 * @return \Stripe\PaymentMethod[]
	 * @throws \Advertikon\Exception
	 */
	public function cards() {
		if ( $this->a->do_cache ) {
			$cached = $this->a->cache->get( $this->ocId() . '_card_all' . ( $this->a->is_live() ? '_live' : '_test' ) );
			if ( !is_null( $cached ) ) return $cached;
		}

		try {
			$this->setAccount();
			$list = \Stripe\PaymentMethod::all( [ 'customer' => $this->stripeId(), 'type' => 'card', ] );

		} catch ( Api $e ) {
			$this->delete();
			$this->a->error( $e );
		}

		$this->a->log( sprintf( " %s Stripe Card(s) was fetched", $list->count() ) );

		if ( $this->a->do_cache ) {
			$this->a->cache->set( $this->ocId() . '_card_all' . ( $this->a->is_live() ? '_live' : '_test' ), $list, 60 );
		}

		return $list->data;
	}

	private function flushCardCache() {
		$this->a->cache->delete( $this->ocId() . '_card_all' . ( $this->a->is_live() ? '_live' : '_test' ) );
	}

	/**
	 * @param $methodId
	 * @return \Stripe\PaymentMethod
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function card( $methodId ) {
		$this->setAccount();
		return \Stripe\PaymentMethod::retrieve( $methodId );
	}

	/**
	 * @param $methodId
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function deleteCard( $methodId ) {
		$customer = $this->fetch();
		$cards = $this->cards();

		if( $customer->subscriptions->total_count > 0 && count( $cards ) < 2 ) {
			throw new Exception(
				$this->a->__(
					'You may not delete the only payment source until you have an active subscription'
				)
			);
		}

		$card = $this->card( $methodId );
		$card->detach();
		$this->flushCardCache();
	}

	/**
	 * @param null $methodId
	 * @return string
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function defaultPaymentMethod( $methodId = null ) {
		if ( is_null( $methodId ) ) {
			$customer = $this->fetch();
			return $customer->invoice_settings->default_payment_method;
		}

		$this->setAccount();
		\Stripe\Customer::update( $this->stripeId(), [
			'invoice_settings' => [
				'default_payment_method' => $methodId
			]
		] );

		return null;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * @throws \Advertikon\Exception
	 */
	private function canBeSaved() {
		return trim( $this->ocId ) && trim( $this->stripeId );
	}

	private function setData( array $data ) {
		$this->id        = isset( $data['id'] ) ? $data['id'] : null;
		$this->ocId      = isset( $data['oc_customer_id'] ) ? $data['oc_customer_id'] : null;
		$this->stripeId  = isset( $data['stripe_id'] ) ? $data['stripe_id'] : null;
		$this->dateAdded = isset( $data['date_added'] ) ? $data['date_added'] : null;
		$this->account   = isset( $data['account'] ) ? $data['account'] : null;
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	private function saveNew() {
		$query = Sql::insert( self::TABLE, true );
		$query->set( [
			'oc_customer_id' => $this->ocId,
			'stripe_id'      => $this->stripeId,
			'date_added'     => date( 'Y-m-d H:i:s'),
			'date_modified'  => date( 'Y-m-d H:i:s'),
			'is_test'        => $this->getMode(),
			'account'        => StripeAccount::getCurrentAccount()->getCode(),
		] );

		$query->run();
		$this->refresh();
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	private function refresh() {
		$res = Sql::select(self::TABLE, true )
			->where('oc_customer_id')->equal($this->ocId )
			->where('stripe_id')->equal($this->stripeId )
			->where('is_test')->equal($this->getMode())->end()
			->run();

		$this->setData( $res[0] );
	}
}