<?php
/**
 * Advertikon Stripe Class
 * @author Advertikon
 * @package Advertikon
 * @version 5.0.44  
 */

namespace Advertikon\Stripe;

use Advertikon\Element\Bootstrap\Button;
use Advertikon\Element\Bootstrap\InputGroup;
use Advertikon\Element\Checkbox;
use Advertikon\Element\Div;
use Advertikon\Element\Element;
use Advertikon\Element\Image;
use Advertikon\Element\Input;
use Advertikon\Element\Input\Number;
use Advertikon\Element\Label;
use Advertikon\Element\Password;
use Advertikon\Element\Select;
use Advertikon\Element\Slider;
use Advertikon\Element\Span;
use Advertikon\Setting;
use Advertikon\Stripe\PaymentOption\Card;
use Advertikon\Stripe\PaymentOption\Vendor;

class Extended {
	/**
	 * @param $data
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
	public static function order( &$data, &$locale ) {
		$a = Advertikon::instance();
		$order = $a->get_order_model();
		$order_id = $a->requestParam( 'order_id' );
		$token = version_compare( VERSION, '3', '>=' ) ?
            $a->session_get( 'user_token' ) : $a->session_get( 'token' );

		$order_info = $order->getOrder( $order_id );
		$aOrder = Order::get( $order_id );

		if ( !$aOrder->exists() ) {
			throw new Exception( $a->__( 'No data to display' ) );
		}

		if ( !isset( $order_info['order_id'] ) ) {
            throw new Exception( $a->__( 'Order is missing' ) );
		}

		$charge = $aOrder->fetch();
		$paymentOption = PaymentOption::get( $charge->payment_method_details->type );
        $a->log( $charge );

        if ( is_null( $charge ) ) {
            throw new Exception('Charge is missing' );
        }

		$amount            = OrderPrice::fromCents( $charge->amount, $charge->currency );
		$refunded          = OrderPrice::fromCents( $charge->amount_refunded, $charge->currency );
		$available_amount  = $amount - $refunded;

		$data = array_merge( $data, array(
			'order' 			  => $charge,
			'order_id'			  => $order_id,
			'token'				  => $token,
			'amount'              => $amount,
			'amount_formatted'    => $a->currency->format( $amount, strtoupper( $charge->currency ), 1, true ),
			'refunded'            => $refunded,
			'refunded_formatted'  => $a->currency->format( $refunded, strtoupper( $charge->currency ), 1, true ),
			'operatable_amount'   => $available_amount,
			'refund_reason'       => self::renderRefundReason(),
			'currency'            => strtoupper( $charge->currency ),
			'releaseTime'         => Order::captureReleaseSpan( $charge )
		) );

		$locale = array_merge( $locale, [
			'oc_history_url'    => $a->u( 'sale/order/history', [ 'order_id' => $order_id ] ),
			'history_url'	    => $a->u()->catalog_url() . 'index.php?route=api/order/history&order_id=' . $order_id,
			'order_id'          => $order_id,
			'amount'            => $amount,
			'operatable_amount' => $available_amount,
			'minAmountAlert'    => $a->__( "Amount should be greater than zero" ),
			'current_status'	=> $order_info['order_status_id'],
			'order_url'			=> $a->u()->url( 'ordershow', [ 'order_id' => $order_id ] ),
		] );

        self::getSourceInfo( $charge, $data );
        self::getRefunds( $charge, $data );
        self::getCaptureButton( $charge, $data );

        if ( $paymentOption->supportsRefund() ) {
	        self::getRefundButton( $charge, $data );
        }

        self::getRefreshButton( $data );
	}

    /**
     * @param \Stripe\Charge $charge
     * @param $data
     * @throws \Advertikon\Exception
     */
	private static function getSourceInfo( \Stripe\Charge $charge, &$data ) {
        if ( $charge->payment_method_details ) {
            $data = array_merge( $data, self::getCardInfo( $charge->payment_method_details ) );
        }
    }

	/**
	 * @param \Stripe\PaymentMethod $method
	 * @return array
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
    static public function getCardInfo( \Stripe\StripeObject $method ) {
		$data = [];

	    if ( $method->card ) {
		    $card = $method->card;
		    $cardOption = new Card();
		    $vendors = $cardOption->vendors();

		    $data['source_type'] = 'card';
		    $data['id']          = $method->id;
		    $data['last4']       = $card->last4;
		    $data['exp_month']   = str_pad( $card->exp_month, 2, '0', STR_PAD_LEFT );
		    $data['exp_year']    = $card->exp_year;

		    /** @var Vendor $vendor */
		    foreach( $vendors as $vendor ) {
			    if ( $vendor->getTag() === $card->brand ) {
				    $data['brand']       = $vendor->getTitle();
				    $data['brand_image'] = (new Image( $vendor->getIcon() ))->style()->width('50px')->stop();
			    }
		    }

		    if ( $card->checks ) {
			    $checks = $card->checks;
			    $data['address_line1_check'] = $checks->address_line1_check;
			    $data['address_zip_check']   = $checks->address_postal_code_check;
			    $data['cvc_check']           = $checks->cvc_check;
		    }

		    if ( $card->three_d_secure && $card->three_d_secure->succeeded ) {
			    $data['is_three_d_secure'] = true;
		    }

	    } else {
	    	$option = PaymentOption::get( $method->type );

	    	if ( !is_null($option ) ) {
			    $data['source_type'] = $option->getId();
			    $data['brand']       = $option->getTitle();
			    $data['brand_image'] = (new Image( $option->iconUrl() ))->style()->width('50px')->stop();
		    }
	    }

	    return $data;
    }

    static private function getRefunds( \Stripe\Charge $charge, &$data ) {
	    $refunds = [];

	    if ( $charge->refunds ) {
	        /** @var \Stripe\Refund $refund */
            foreach( $charge->refunds->data as $refund ) {
	            $refunds[] = [
	                'created'             => $refund->created,
                    'amount'              => OrderPrice::fromCents( $refund->amount, $refund->currency ),
                    'currency'            => $refund->currency,
                    'reason'              => $refund->reason,
                    'balance_transaction' => $refund->balance_transaction,
                ];
            }
        }

	    $data['refunds'] = $refunds;
    }

    /**
     * @param \Stripe\Charge $charge
     * @param $data
     * @throws \Advertikon\Exception
     */
    static private function getCaptureButton( \Stripe\Charge $charge, &$data ) {
	    $a = Advertikon::instance();
	    $ret = new InputGroup();
	    $ret->getClass( 'input-group col-sm-10 col-sm-offset-2 adk-input' );
	    $ret->addonBefore( strtoupper( $charge->currency ) );
	    $ret->element(
	        (new Number())
                ->id("adk-capture")
                ->getClass()->add('form-control')
                ->attributes()->set(['placeholder' => $a-> __( 'Amount to capture' )])
                ->value($data['operatable_amount'])
        );
	    $ret->addonAfter(
	        (new Button())
                ->isDefault()
		        ->textBefore( $a->__('Capture' ) )
		        ->icon( 'fa-play' )
		        ->url( $a->u()->url('capture') )
                ->id('adk-capture-button')
		        ->attributes()->set( [
			        'data-confirm'  => $a->__( 'Do capture' ),
			        'data-order-id' => $data['order_id'],
			        'data-currency' => $data['currency'],
			        'data-status'   => Setting::get( 'status_captured', $a ),
			        'data-history'  => $a->__( 'Capture of' ),
		        ] )
        );

	    $data['capture_button'] = $ret;
    }

	/**
	 * @param \Stripe\Charge $charge
	 * @param $data
	 * @throws \Advertikon\Exception
	 */
	static private function getRefundButton( \Stripe\Charge $charge, &$data ) {
		$a = Advertikon::instance();
		$ret = new InputGroup();
		$ret->getClass( 'input-group col-sm-10 col-sm-offset-2 adk-input' );
		$ret->addonBefore( strtoupper( $charge->currency ) );
		$ret->element(
			(new Number())
				->id("adk-refund")
				->getClass()->add('form-control')
				->attributes()->set(['placeholder' => $a-> __( 'Amount to refund' )])
				->value($data['operatable_amount'])
		);
		$ret->addonAfter(
			(new Button())
				->isDefault()
				->textBefore( $a->__('Refund' ) )
				->icon( 'fa-play' )
				->url( $a->u()->url('refund') )
				->id('adk-refund-button')
				->attributes()->set( [
					'data-confirm'  => $a->__( 'Do refund' ),
					'data-order-id' => $data['order_id'],
					'data-currency' => $data['currency'],
					'data-status'   => Setting::get( 'status_voided', $a ),
					'data-history'  => $a->__( 'Refund of' ),
				] )
		);

		$data['refund_button'] = $ret;
	}

	/**
	 * @param $data
	 * @throws \Advertikon\Exception
	 */
	static private function getRefreshButton( &$data ) {
		$a = Advertikon::instance();

		$data['refresh_button'] = ( new Button( $a->__( 'Refresh' ) ) )
			->isPrimary()
			->url( $a->u()->url( 'refresh' ) )
			->icon('fa-refresh' )
			->id('adk-refresh');
	}

    /**
     * @return string|Element
     * @throws \Advertikon\Exception
     * @throws \Exception
     */
	public static function cardLogos() {
		$a = Advertikon::instance();
		$doShow  = Setting::get( 'card_images_show', $a, false );
		$width   = Setting::get( 'card_images_width', $a, Slider::def(null,null,100, '%' ) );
		$padding = Setting::get( 'card_images_padding', $a, Slider::def(null,null,0) );
		/** @var Card $cardMethod */
		$cardMethod = PaymentOption::get( PaymentOption::CARD );

		if ( !$doShow || is_null( $cardMethod ) ) {
			return '';
		}

		$div = (new Div())
            ->getClass()->add('adk-cards-set')
            ->style()->width( Slider::getMax( $width ) )
            ->stop();

        $vendors = $cardMethod->enabledVendors();

		foreach( $vendors as $vendor ) {
		    $div2 = (new Div(new Image( $vendor->getIcon() )))
                ->getClass( 'adk-cards-set-item' )->add( $vendor->getTag() )
                ->style()
                ->padding()
                    ->left( Slider::getMax( $padding ) )
                    ->right( Slider::getMax( $padding ) )
                        ->stop()
                ->stop();

		    $div->children( $div2 );
		}

		return $div;
	}

	/**
	 * @return Element|string
	 * @throws \Advertikon\Exception
	 */
	public static function cardInputs() {
		$a      = Advertikon::instance();
		$layout = Setting::get( 'form_layout', $a );
		$ret = [];

		if ( $layout === 'one_line' ) {
			return Card::oneLineLayoutInputs();
		}

		$wrapper = new Div();

		$wrapper->children( (new Div())->id("card-number" ) );
		$div = (new Div())->id( "card-expiry" );

		if ( $layout === 'two_line' ) {
			$div->getClass( 'two-lines-input' );
		}

		$wrapper->children( $div );
		$div = (new Div())->id( "card-cvv" );

		if ( $layout === 'two_line' ) {
			$div->getClass( 'two-lines-input' );
		}

		$wrapper->children( $div );
		return $wrapper;
	}

    /**
     * @return string|Element
     * @throws \Advertikon\Exception
     * @throws \Exception
     */
	public static function rememberMe() {
		$a = Advertikon::instance();

		if ( Setting::get('remember_me', $a, false ) && $a->customer->isLogged() ) {
			return (new Div(
				new Div(
					(new Checkbox())->id( 'remember-me' )->style(PaymentForm::inputCss())->stop(),
					(new Label( $a-> __( 'caption_form_rerember_me' ) ))->forId( 'remember-me' ),
					(new Div($a->__( 'caption_form_rerember_me_description' )))
						->style()->fontSize('0.8em' )->lineHeight('1.3em')->stop()
				)
			))->getClass()->add( 'adk-form-group' )->id( 'remember-me-wrapper' );
		}

		return '';
	}

	/**
	 * @param array $data
	 * @return string|Element
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
	public static function savedCards() {
		$a = Advertikon::instance();

		if ( !Setting::get( 'remember_me', $a ) ) {
			return '';
		}

		$savedCards = self::savedCardsFetch();

		if ( !$savedCards ) {
			return '';
		}

		$select = (new Select())
			->getClass()->add( 'adk-form-control')
			->id('saved-card')
			->style(PaymentForm::inputCss())
			->stop();

		$div = (new Div(
			(new Div(
				$select
			))->style()->overflow('hidden')->position('relative')->stop()
		))->getClass()->add( 'adk-form-group' )->id( 'saved-cards-wrapper' );

		$select->option( 0, $a->__( 'caption_form_select_card' ) );

		foreach ( $savedCards as $paymentOption ) {
			$card = $paymentOption->card;

			$option = (new \Advertikon\Element\Option(
				$paymentOption->id,
				ucfirst( $card->brand ) . " **** {$card->last4}" .
                ' ( ' . str_pad( $card->exp_month, 2, '0', STR_PAD_LEFT ) .
				' / ' . $card->exp_year . ' )'
			) )->attributes()->set( 'data-image', self::brandImage( $card->brand ) );

			$select->option( $option );
		}

		return $div;
	}

	/**
	 * @param $brand
	 * @return string
	 * @throws \Advertikon\Exception
	 */
	static private function brandImage( $brand ) {
		$name = str_replace( ' ', '', strtolower( $brand ) );

		foreach( (new Card())->vendors() as $vendor ) {
			if ( $vendor->getTag() === $name ) {
				return $vendor->getIcon();
			}
		}

		if ( is_file( PaymentOption::$iconFolder . 'no-image.svg' ) ) {
			return PaymentOption::$iconFolder . 'no-image.svg';
		}

		return '';
	}

	/**
	 * @param null $customerId
	 * @return \Stripe\PaymentMethod[]
	 * @throws \Advertikon\Exception
	 */
	static private function savedCardsFetch( $customerId = null ) {
		$customer = Customer::get( $customerId );
		return $customer->exists() ? $customer->cards() : [];
	}

    /**
     * @return string
     * @throws \Advertikon\Exception
     * @throws \Exception
     */
	public static function image() {
		$a = Advertikon::instance();
		$config_image = (int)Setting::get( 'form_image', $a );
        $div = new Div();

		if ( $config_image !== Advertikon::FORM_IMAGE_NONE ) {
			$image_border_radius = Setting::get( 'form_image_border_radius', $a, [ [], [ 'value' => 0, 'units' => 'px' ] ] );
			$image_width         = Setting::get( 'form_image_width', $a, [ [],  [ 'value' => 0, 'units' => 'px' ] ] );
			$image_padding       = Setting::get( 'form_image_padding', $a, [ [],  [ 'value' => 0, 'units' => 'px' ] ] );
			$image_border_width  = Setting::get( 'form_image_border_width', $a, [ [],  ['value' => 0, 'units' => 'px' ] ] );
			$image_top           = Setting::get( 'form_image_top', $a, [ [], [ 'value' => 0, 'units' => 'px' ] ] );
			$image_left          = Setting::get( 'form_image_left', $a, [ [], [ 'value' => 0, 'units' => 'px' ] ] );
			$image_border_color  = Setting::get( 'form_image_border_color', $a, '#ffffff' );
			$url = '';

			switch ( $config_image ) {
				case Advertikon::FORM_IMAGE_LOGO:
					$url = $a->u()->path_to_url( DIR_IMAGE . $a->config->get( 'config_image' ) );
					break;
				case Advertikon::FORM_IMAGE_PRODUCT:
					foreach( $a->cart->getproducts() as $p ) {
						if ( $p['image'] ) {
							$url = $a->u()->path_to_url( DIR_IMAGE . $p['image'] );
							break;
						}
					}
					break;
				case Advertikon::FORM_IMAGE_CUSTOM:
					$url = $a->u()->path_to_url( DIR_IMAGE . Setting::get( 'form_image_custom', $a ) );
					break;
			}

			$image = (new Image( $url ))->style()->width('100%!important' )->stop();
			$div->children( $image );

			$div ->style()
                ->position( 'relative!important' )
                ->borderRadius( Slider::getMax( $image_border_radius ) )->important()->stop()
                ->width( Slider::getMax( $image_width ) )
                ->border()
                    ->width( Slider::getMax( $image_border_width ) )
                    ->color( $image_border_color )
                    ->style('solid' )->important()->stop()
                ->left( Slider::getMax( $image_left ) . '!important' )
                ->top( Slider::getMax( $image_top ) . '!important' )
                ->padding( Slider::getMax( $image_padding ) )->important()->stop()
                ->overflow( 'hidden!important' )
                ->backgroundColor( 'inherit!important' );
		}

		return $div;
	}

	/**
	 * @param array $data
	 * @return string|Element
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
	public static function cardholderName() {
		$a = Advertikon::instance();
		$name = '';

		if( isset(
			$a->session->data['payment_address']['firstname'],
			$a->session->data['payment_address']['lastname'] )
		) {
			$name = "{$a->session->data['payment_address']['firstname']} {$a->session->data['payment_address']['lastname']}";
		}

		if ( !Setting::get( 'card_name', $a ) ) return  '';

		return (new Div(
			(new Input())
				->id('cc-name')
				->getClass()->add('adk-form-control')
				->attributes()->set('placeholder', $a->__( 'caption_form_cardholder_name' ))
				->style(PaymentForm::inputCss())->stop()
				->value($name)
		))->getClass()->add('adk-form-group');
	}

	/**
	 * @return \Advertikon\Renderer|string
	 * @throws \Advertikon\Exception
	 */
	protected static function renderRefundReason() {
		$a = Advertikon::instance();

		return (new Select())
			->getClass()->add( 'refund-reason form-control' )
			->id( 'refund-reason' )
			->option( [
				0                                   => $a->__( 'Select a reason' ),
				Advertikon::REFUND_REASON_DUPLICATE => $a->__( 'Duplicated' ),
				Advertikon::REFUND_REASON_CUSTOMER  => $a->__( 'Requested by customer' ),
				Advertikon::REFUND_REASON_FRAUD     => $a->__( 'Fraudulent' ),
			] );
	}
}