<?php
/**
 * Advertikon Stripe Order Class
 * @author Advertikon
 * @package Stripe
 * @version 5.0.44  
 */

namespace Advertikon\Stripe;


use Advertikon\Setting;
use Advertikon\Sql;
use PhpOffice\PhpSpreadsheet\Calculation\DateTime;

class Order {

	/** @var Advertikon */
	private $a;

	private $id;
	private $account;
	private $code;

	const TABLE = 'advertikon_stripe_order';

	/**
	 * @param $code
	 * @return bool
	 * @throws \Advertikon\Exception
	 */
	static public function is_processed( $code ) {
		return count( Sql::select( self::TABLE )->where('code')->equal($code)->end()->run() ) > 0;
	}

	/**
	 * @param $id
	 * @param $code
	 * @return Order
	 * @throws \Advertikon\Exception
	 */
	static function create( $id, $code ) {
	    $ret = new self( $id, $code, StripeAccount::getCurrentAccount()->getCode() );
	    $ret->save();
	    return $ret;
    }

    /**
     * @param $orderId
     * @return Order
     * @throws \Advertikon\Exception
     */
	static public function get( $orderId ) {
	    $query = Sql::select(self::TABLE, true )->where('id')->equal($orderId)->end()->run();

	    if ( !$query ) {
	        return new self();
        }

	    return new self( $query[0] );
    }

    /**
     * @param $code
     * @return Order
     * @throws \Advertikon\Exception
     */
    static public function getByCode( $code ) {
        $query = Sql::select(self::TABLE, true )->where('code')->equal($code)->end()->run();

        if ( !$query ) {
            return new self();
        }

        return new self( $query[0] );
    }

	/**
	 * @param \Stripe\Charge $charge
	 * @return \DateTime
	 * @throws \Exception
	 */
    static function captureReleaseDay( \Stripe\Charge $charge ) {
    	return (new \DateTime)->setTimestamp( $charge->created )->add( \DateInterval::createFromDateString( '+7 day' ) );
    }

	/**
	 * @param \Stripe\Charge $charge
	 * @return String
	 * @throws \Exception
	 */
    static function captureReleaseSpan( \Stripe\Charge $charge ) {
    	$a = Advertikon::instance();
    	$diff = self::captureReleaseDay( $charge )->diff( new \DateTime );

    	if ( $diff->format( '%a' ) >= 1 ) {
    		return $a->__( 'Releases in %s day(s)', $diff->format( '%a' ) );

	    } else {
		    return $a->__( 'Releases in %s hour(s)', $diff->format( '%h' ) );
	    }
    }

    /**
     * Order constructor.
     * @param null $id
     * @param null $code
     * @param null $account
     * @throws \Advertikon\Exception
     */
	public function __construct( $id = null, $code = null, $account = null ) {
		$this->a = Advertikon::instance();

		if ( !is_null( $id ) && is_array( $id ) ) {
			$this->id = $id['id'];
			$this->code = $id['code'];
			$d = json_decode( $id['data'], true );

			if ( isset( $d['account'] ) ) {
			    $this->account = $d['account'];
            }

		} else {
		    $this->id = $id;
		    $this->code = $code;
		    $this->account = $account;
        }
	}

	public function getId() {
		return $this->id;
	}

	public function getCode() {
		return $this->code;
	}

	public function exists() {
	    return !is_null( $this->id ) && !is_null( $this->code );
    }

	/**
	 * @return \Stripe\Charge|null
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
    public function fetch() {
	    $account = $this->account ?: StripeAccount::getFallbackCode();

	    if ( is_null($account ) ) {
		    throw new Exception("Undefined account" );
	    }

	    $this->a->setAccount( $account );

        try {
        	$this->a->setAccount( $account );
            return \Stripe\Charge::retrieve($this->code);

        } catch ( \Exception $e) {
            // TODO: delete
        }

        return null;
    }

    /**
     * @return int|\stdClass
     * @throws \Advertikon\Exception
     */
	public function save() {
	    return Sql::insert(self::TABLE, true)->set([
	        'id'   => $this->id,
            'code' => $this->code,
            'data' => json_encode( ['account' => $this->account ] )
        ])->run();
    }

	/**
	 * Capture stripe charge
	 * @param \Stripe\Charge $charge Charge object
	 * @param Integer $amount Amount (in cents) to be captured
	 * @param Integer $order_id Order ID, which charge belongs to
	 * @param bool $notify
	 * @param bool $override
	 * @return Object
	 * @throws Exception On charge fetching error
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
	public function capture( $amount, $notify = false, $override = false ) {
		$charge = $this->fetch();
		//$order = \Advertikon\Order::get_by_id( $this->getId(), $this->a );

		$data = [
//			'amount' => $amount,
			'amount_to_capture' => OrderPrice::toCents( $amount, $charge->currency )
		];

		if ( $statement_descriptor = Setting::get( 'statement_descriptor', $this->a, null ) ) {
			$data['statement_descriptor'] = $statement_descriptor;
		}

		/** @var \Stripe\PaymentIntent $intent */
		$intent = \Stripe\PaymentIntent::retrieve( $charge->payment_intent );
		$intent->capture( $data );
		$charge = \Stripe\Charge::retrieve( $charge->id );

		$this->a->log(
			sprintf(
				'Charge for the order %s was captured on sum of %s %s',
				$this->id,
				OrderPrice::fromCents( $amount, $charge->currency ),
				$charge->currency
			)
		);

		return $charge;
	}

	/**
	 * @param $amount
	 * @param string $reason
	 * @param bool $notify
	 * @param bool $override
	 * @return \Stripe\Charge|null
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function refund( $amount, $reason = '', $notify = false, $override = false ) {
		$charge = $this->fetch();

		$data = array(
			'amount' => OrderPrice::toCents( $amount, $charge->currency ),
			'charge' => $this->getCode(),
		);

		if ( $reason ) {
			$data['reason'] = $reason;
		}

		$refund = \Stripe\Refund::create( $data );
		$this->a->log( "Order #{$this->getId()} was refunded on sum of $amount {$charge->currency}" );

		return $charge;
	}

	/**
	 * @param bool $notify
	 * @param bool $override
	 * @throws \Advertikon\Exception
	 */
	public function setStatusCapture( $notify = true, $override = true ) {
		$this->setStatus( Setting::get( 'status_captured', $this->a ), $notify, $override );
	}

	/**
	 * @param bool $notify
	 * @param bool $override
	 * @throws \Advertikon\Exception
	 */
	public function setStatusAuthorize($notify = true, $override = true) {
		$this->setStatus( Setting::get( 'status_authorized', $this->a ), $notify, $override );
	}

	/**
	 * @param bool $notify
	 * @param bool $override
	 * @throws \Advertikon\Exception
	 */
	public function setStatusRefund($notify = true, $override = true) {
		$this->setStatus( Setting::get( 'status_voided', $this->a ), $notify, $override );
	}

	/**
	 * Mark order as authorized
	 * @param $orderStatusId
	 * @param $notify
	 * @param $override
	 * @throws \Advertikon\Exception
	 */
	private function setStatus( $orderStatusId, $notify, $override ) {
		$order_model = $this->a->get_order_model( 'catalog' );
		$currentStatus = $this->getStatus();

		$this->a->log( "Changing order status from $currentStatus => $orderStatusId" );

		if ( $currentStatus == $orderStatusId ) {
			$this->a->log( 'Status is the same, skip' );
			return;
		}

		try {
			$order_model->addOrderHistory( $this->getId(), $orderStatusId, '', $notify, $override );

		} catch ( \Exception $e ) {
			$this->a->error( $e );
			$this->setOrderStatusForce( $orderStatusId );

		} catch ( \Error $e ) {
			$this->a->error( $e );
			$this->setOrderStatusForce( $orderStatusId );
		}
	}

	/**
	 * @throws \Advertikon\Exception
	 */
	private function getStatus() {
		$query = Sql::select('order', true )->field('order_status_id')
			->where('order_id')->equal($this->getId())->end()->run();

		return isset( $query[0]['order_status_id'] ) ? $query[0]['order_status_id'] : null;
	}

	/**
	 * @param $statusId
	 * @throws \Advertikon\Exception
	 */
	private function setOrderStatusForce( $statusId ) {
		$this->a->log( "Forcing status of order {$this->getId()} to $statusId" );
		Sql::update( 'order' )->set('order_status_id', $statusId )
			->where('order_id')->equal($this->getId())->end()->run();
	}
}