<?php


namespace Advertikon\Stripe;


use Advertikon\Setting;
use Advertikon\Sql;

class OrderPrice {
    /** @var Advertikon */
    private $a;

    static private $nc = [ 'BIF', 'CLP', 'DJF', 'GNF', 'JPY', 'KMF', 'KRW', 'MGA', 'PYG', 'RWF', 'VND', 'VUV', 'XAF',
        'XOF', 'XPF' ];

    static public function toCents( $amount, $currency ) {
        $amount = (float)$amount;

        if ( in_array( strtoupper( $currency ), self::$nc ) ) {
            return (int)$amount;
        }

        return (int)round( $amount * 100 );
    }

    /**
     * Convert cents to amount
     * @param integer
     * @param string
     * @return number
     */
    static public function fromCents( $cents, $currency ){
        if ( in_array( strtoupper( $currency ), self::$nc ) ) {
            return (int)$cents;
        }

        return $cents / 100 ;
    }

    /**
     * @param $price
     * @param bool $inCents
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    static public function price( $price, $inCents = false ) {
        $currency = (new self())->currency();
        $a = Advertikon::instance();

        $value = $a->currency->convert( $price, $a->config->get( 'config_currency' ), strtoupper( $currency ) );

        if ( $inCents ) {
            return self::toCents( $value, $currency );
        }

        return $value;
    }

    /**
     * @param $recurring
     * @param bool $inCents
     * @return int|mixed
     * @throws \Advertikon\Exception
     */
    static public function recurringPrice( $recurring, $inCents = false ) {
        $a = Advertikon::instance();

        $amount = $a->tax->calculate($recurring['recurring']['price'] * $recurring['quantity'],
            $recurring['tax_class_id'], $a->config->get('config_tax') );

        $currency = self::getCurrency();
        $value = $a->currency->convert( $amount, $a->config->get( 'config_currency' ), strtoupper( $currency ) );

        if ( $inCents ) {
            return self::toCents( $value, self::getCurrency() );
        }

        return $value;
    }

    /**
     * @return array
     * @throws \Advertikon\Exception
     */
    static public function recurringIds() {
        $ret = [];

        foreach( Advertikon::instance()->cart->getRecurringProducts() as $product ) {
            $ret[] = $product['cart_id'] ?? $product['key'];
        }

        return $ret;
    }

    /**
     * @return mixed|string
     * @throws \Advertikon\Exception
     */
    static public function getCurrency() {
        $a = Advertikon::instance();
        $sessionCurrency = $a->session->data['currency'];

        if ( !$a->isExtended ) {
            return $sessionCurrency;
        }

        $account = StripeAccount::byCurrency();

        if ( $account->getAccountCurrency() === $sessionCurrency ) {
            return $sessionCurrency;
        }

        if ( Setting::get( 'payment_currency', $a ) == Advertikon::CURRENCY_ORDER ) {
            return $sessionCurrency;
        }

        return $account->getAccountCurrency(); // default Stripe account
    }

    /**
     * OrderPrice constructor.
     * @param $orderId
     * @throws \Advertikon\Exception
     */
    public function __construct() {
        $this->a = Advertikon::instance();
    }

    /**
     * @return array
     * @throws Exception
     */
    public function totals() {
        $taxes = $this->a->cart->getTaxes();
        $totals = [];
        $total = 0;

        if ( version_compare( VERSION, '3', '<' ) ) {
            $this->a->load->model( 'extension/extension' );
            $results = $this->a->model_extension_extension->getExtensions( 'total'  );

        } else {
            $this->a->load->model( 'setting/extension' );
            $results = $this->a->model_setting_extension->getExtensions( 'total' );
        }

        $total_data = array(
            'totals' => &$totals,
            'taxes'  => &$taxes,
            'total'  => &$total
        );

        $this->getTotals( $results, $total_data, $totals, $total, $taxes );
        $sort = [];

        foreach( $totals as $r ) {
            $sort[$r['code']] = $r['sort_order'];
        }

        usort( $results, function( array $a, array $b ) use( $sort ) {
            if ( isset( $sort[ $a['code'] ], $sort[ $b['code'] ] ) ) {
                return $sort[ $a['code'] ] - $sort[ $b['code'] ];
            }

            if ( isset( $sort[ $a['code'] ] ) ) {
                return 1;
            }

            if ( isset( $sort[ $b['code'] ] ) ) {
                return -1;
            }

            return 0;
        } );

        $taxes = $this->a->cart->getTaxes();
        $totals = [];
        $total = 0;
        $this->getTotals( $results, $total_data, $totals, $total, $taxes );

        return $totals;
    }

    private function getTotals( array $results, array &$total_data, array &$totals, &$total, &$taxes ) {
        foreach ( $results as $result ) {
            if ( $this->a->config->get( $result['code'] . '_status' ) || $this->a->config->get( 'total_' . $result['code'] . '_status' ) ) {
                if ( version_compare( VERSION, '2.2', '<' ) ) {
                    $this->a->load->model( 'total/' . $result['code'] );
                    $this->a->{'model_total_' . $result['code']}->getTotal( $totals, $total, $taxes );

                } else if ( version_compare( VERSION, '2.2.0.0', '==' ) ) {
                    $this->a->load->model( 'total/' . $result['code'] );
                    $this->a->{'model_total_' . $result['code']}->getTotal( $total_data );

                } else {
                    $this->a->load->model( 'extension/total/' . $result['code'] );
                    $this->a->{'model_extension_total_' . $result['code']}->getTotal( $total_data );
                }
            }
        }
    }

    private $getMainStoreCurrency;

    private function getMainStoreCurrency() {
        if ( !$this->getMainStoreCurrency ) {
            $this->getMainStoreCurrency = $this->a->db->query( 'select * from ' . DB_PREFIX . 'setting where `key` = "config_currency" and store_id = 0' )->row['value'];
        }

        return $this->getMainStoreCurrency;

    }

    /**
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function totalStore() {
        foreach( $this->totals() as $total ) {
            if ( $total['code'] === 'total' ) {
                return $total['value'];
            }
        }

        throw new Exception( 'No totals found' );
    }

    /**
     * @param bool $inCents
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function total( $inCents = false ) {
        $val = $this->convert( $this->totalStore() ) ;$this->a->log( $val);
        return $inCents ? self::toCents( $val, $this->currency() ) : $val;
    }

    /**
     * @param bool $inCents
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function totalEuro( $inCents = false ) {
        $val = $this->convert( $this->totalStore(), 'EUR' );
        return $inCents ? self::toCents( $val, "EUR" ) : $val;
    }

    /**
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function recurringStore() {
        $total = 0;

        foreach( $this->a->cart->getRecurringProducts() as $product ) {
            if ( !self::isSupportedRecurring( $product['recurring'] ) ) {
                throw new Exception( 'Non-free trials not supported' );
            }

            if ( self::isTrial( $product['recurring'] ) ) {
                continue;
            }

            $total +=  $this->a->tax->calculate($product['recurring']['price'] * $product['quantity'],
                $product['tax_class_id'], $this->a->config->get('config_tax') );
        }

        return $total;
    }

    /**
     * @param bool $inCents
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function recurring( $inCents = false ) {
        $val = $this->convert( $this->recurringStore() );
        return $inCents ? self::toCents( $val, $this->currency() ) : $val;
    }

    /**
     * @param bool $inCents
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function recurringEuro( $inCents = false ) {
        $val = $this->convert( $this->recurringStore(), 'EUR' );
        return $inCents ? self::toCents( $val, "EUR" ) : $val;
    }

    /**
     * @return mixed
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function currency() {
        return self::getCurrency();
    }

    public static function isTrial( array $recurringPlan ) {
        return (int)$recurringPlan['trial_duration'] > 0 && (int)$recurringPlan['trial_cycle'] > 0 && $recurringPlan['trial'];
    }

    static public function isSupportedRecurring( array $recurringPlan ) {
        return !self::isTrial( $recurringPlan ) || $recurringPlan['trial_price'] == 0;
    }

    /**
     * @param $value
     * @param null $to
     * @return float|int
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function convert( $value, $to = null ) {
        if ( is_null( $to ) ) {
            $to = $this->currency();
        }

        return $this->a->currency->convert( $value, $this->getMainStoreCurrency(), $to );
    }

    /**
     * @return bool
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function isDifferentCurrency() {
        return isset( $this->a->session->data['currency'] ) && $this->a->session->data['currency'] !== $this->currency();
    }
}