<?php
/**
 * Advertikon Stripe Option Class
 * @author Advertikon
 * @package Advertikon
 * @version 5.0.44      
 */

namespace Advertikon\Stripe;


use Advertikon\Element\Bootstrap\Accordion;
use Advertikon\Element\Bootstrap\FormGroup;
use advertikon\element\Div;
use Advertikon\Element\FancyCheckBox;
use Advertikon\Element\Italic;
use Advertikon\Element\Slider;
use Advertikon\Setting;

/**
 * Class Option
 * @package Advertikon\Stripe
 * @method array next_year
 * @method array payment_option
 * @method array stripe_account
 * @method array payment_system
 * @method array input_appearance
 */
abstract class PaymentOption {
    const ALIPAY         = 'alipay';
    const BANCONTACT     = 'bancontact';
    const CARD           = 'card';
    const EPS            = 'eps';
    const GIROPAY        = 'giropay';
    const IDEAL          = 'ideal';
    const MULTIBANKO     = 'multibanco';
    const PAYMET_REQUEST = 'payment_request';
    const SOFORT         = 'sofort';
    const WECHAT         = 'wechat';
    const P24            = 'p24';

    protected $locales = ['auto', 'ar', 'bg', 'cs', 'da', 'de', 'el', 'et', 'en', 'es', 'fi', 'fr', 'he', 'id', 'it',
        'ja', 'lt', 'lv', 'ms', 'nb', 'nl', 'pl', 'pt-BR', 'pt', 'ru', 'sk', 'sl', 'sv', 'zh'];

    abstract function icon();

    /** @var \Advertikon\Stripe\Advertikon */
    protected $a;

    static public $iconFolder;

    /**
     * @return PaymentOption[]
     * @throws \Advertikon\Exception
     */
    static function getAll() {
        return [
            new PaymentOption\Card(),
            new PaymentOption\PaymentRequest(),
            new PaymentOption\Alipay(),
            new PaymentOption\Bancontact(),
            new PaymentOption\Eps(),
            new PaymentOption\Giropay(),
            new PaymentOption\Ideal(),
            new PaymentOption\Multibanco(),
            new PaymentOption\Sofort(),
            new PaymentOption\WeChat(),
            new PaymentOption\P24(),
        ];
    }

    /**
     * @return array
     * @throws \Advertikon\Exception
     */
    static public function getAllNames() {
        $ret = [];

        foreach( self::getAll() as $option ) {
            $ret[$option->getId()] = $option->getTitle();
        }

        return $ret;
    }
    /**
     * @return PaymentOption[]
     * @throws \Advertikon\Exception
     */
    static function getAllExtended() {
        return array_filter( self::getAll(), function(PaymentOption $p) {return $p->isExtended();} );
    }

    /**
     * @return array
     * @throws \Advertikon\Exception
     */
    static function getAllEnabled() {
        return array_filter( self::getAllExtended(), function(PaymentOption $p) {return $p->isEnabled();} );
    }

    /**
     * @param $tag
     * @return PaymentOption|null
     * @throws \Advertikon\Exception\
     * @throws \Advertikon\Exception
     */
    static function get( $tag ) {
        /** @var PaymentOption $item */
        foreach ( self::getAllEnabled() as $item ) {
            if ( $item->getId() === $tag ) return $item;
        }

        return null;
    }

    /**
     * @throws \Advertikon\Exception
     */
    static public function adminContent() {
        $accordion = new Accordion();

        foreach( self::getAllExtended() as $p ) {
            $accordion->card()
                ->header( $p->title )
                ->image( $p->iconUrl() )
                ->body( $p->getAdminSettings() );
        }

        return $accordion;
    }

    /**
     * PaymentOption constructor.
     * @throws \Advertikon\Exception
     */
    public function __construct() {
        $this->a = \Advertikon\Stripe\Advertikon::instance();
        self::$iconFolder = DIR_IMAGE . 'catalog/advertikon/advertikon_stripe/';
    }

    public function getId() {
        return $this->tag;
    }

    public function getStatusTag() {
        return "{$this->tag}_status";
    }

	public function getShowIconTag() {
		return "{$this->tag}_show_icon";
	}

    public function isExtended() {
        return $this->a->isExtended;
    }

    public function getTitle() {
    	return $this->title;
    }

    public function supportsRefund() {
    	return true;
    }

    public function supportsRecurring() {
        return false;
    }

    protected function isEnabled() {
        return Setting::get( $this->getStatusTag(), $this->a, true );
    }

	public function isShowIcon() {
		return Setting::get( $this->getShowIconTag(), $this->a, true );
	}

	public function isCard() {
    	return $this->tag === PaymentOption::CARD;
	}

    /**
     * @return \Advertikon\Element\Element
     * @throws \Exception
     */
	public function formHeader() {
        $paymentOptionsPadding = $this->a->config( 'form_header_padding', Slider::def( null, null, 5 )  );

    	return (new Italic( $this->title ))
            ->id( $this->getId() )
            ->attributes()->set( 'data-code', $this->getId() )
            ->getClass()->add('payment-option-header-item' )
            ->style()
            ->padding()
                ->left( Slider::getMax( $paymentOptionsPadding ) )
                ->right( Slider::getMax( $paymentOptionsPadding ) )
                ->top('5px')
                ->bottom('5px' )
                ->important()
                ->stop()
            ->stop();
	}

    /**
     * @return \Advertikon\Element\Element
     */
	public function formBody() {
    	return (new Div())->getClass( $this->getId() )->add("adk-method-item-body");
	}

    /**
     * @return string
     * @throws \Advertikon\Exception
     */
    public function iconUrl() {
        return $this->a->u()->path_to_url( $this->icon() );
    }

    public function getAdminSettings() {
        $wrapper = new Div();

        $status = (new FormGroup())
            ->label( $this->a->__( 'Status' ) )
            ->element( (new FancyCheckBox())
                ->value( $this->isEnabled() )
                ->name( $this->getStatusTag(), $this->a ) );

        $wrapper->children( $status );

	    $status = (new FormGroup())
		    ->label( $this->a->__( 'Show icon' ) )
		    ->element( (new FancyCheckBox())
			    ->value( $this->isShowIcon() )
			    ->name( $this->getShowIconTag(), $this->a ) );

	    $wrapper->children( $status );

        return $wrapper;
    }

	/**
	 * Returns list of available payment options
	 * @return array
	 * @throws \Advertikon\Exception
	 */
	public function get_payment_option() {
		return array(
			Advertikon::PAYMENT_AUTHORIZE         => ADK( __NAMESPACE__ )->__( 'Authorize' ),
			Advertikon::PAYMENT_AUTHORIZE_CAPTURE => ADK( __NAMESPACE__ )->__( 'Capture' ),
			Advertikon::PAYMENT_FRAUD_CHECK       => ADK( __NAMESPACE__ )->__( 'Authorize if fraud' ),
		);
	}
}
