<?php


namespace Advertikon\Stripe\PaymentOption;


use Advertikon\Element\Bootstrap\FormGroup;
use Advertikon\Element\Div;
use Advertikon\Element\FancyCheckBox;
use Advertikon\Element\Hr;
use Advertikon\Element\Image;
use Advertikon\Setting;
use Advertikon\Stripe\Advertikon;
use Advertikon\Stripe\Customer;
use Advertikon\Stripe\Exception;
use Advertikon\Stripe\Extended;
use Advertikon\Stripe\OrderPrice;
use Advertikon\Stripe\PaymentForm;
use Advertikon\Stripe\PaymentOption;

class Card extends PaymentOption {
    public $tag = PaymentOption::CARD;
    public $title = 'Card';

    public function __construct() {
        parent::__construct();
        $this->title = $this->a->__('Card' );
    }

	/**
	 * @return Vendor[]
	 * @throws \Advertikon\Exception
	 */
    public function vendors() {
        return [
            new Vendor('amex', 'American Express','amex.svg' ),
            new Vendor( 'diners', 'Diners Club', 'diners.svg' ),
            new Vendor( 'jcb', 'JCB', 'jcb.svg' ),
            new Vendor( 'visa', 'Visa', 'visa.svg' ),
            new Vendor( 'mastercard', 'MasterCard', 'mastercard.svg' ),
            new Vendor( 'discover', 'Discover', 'discover.svg' ),
        ];
    }

	/**
	 * @return array
	 * @throws \Advertikon\Exception
	 */
    public function enabledVendors() {
        return array_filter( $this->vendors(), function(Vendor $v) { return $v->isEnabled(); } );
    }

	/**
	 * @throws \Advertikon\Exception
	 */
    public function enabledVendorsCodes() {
    	$ret = [];

    	/** @var Vendor $vendor */
	    foreach( $this->enabledVendors() as $vendor ) {
    		$ret[] = $vendor->getTag();
	    }

	    return $ret;
    }

    public function isExtended() {
        return true;
    }

    public function supportsRecurring() {
        return true;
    }

    function icon() {
        return PaymentOption::$iconFolder . 'visa.svg';
    }

    /**
     * @return Div|void
     * @throws \Advertikon\Exception
     */
    public function getAdminSettings() {
        $wrapper = parent::getAdminSettings();
        $div = new Div();

        $group = (new FormGroup())
            ->label( $this->a->__( 'Vendors' ) )
            ->element( $div );

        $wrapper->children( $group );

        /**
         * @var $vendor Vendor
         */
        foreach ( $this->vendors() as $vendor ) {
            $line = new FormGroup();
            $line->label( (new Image( $vendor->getIcon(), $vendor->getTitle() ))->style()->width( '50px' )->stop() );
            $line->element( (new FancyCheckBox())
                ->value( $vendor->isEnabled() )
                ->name( $vendor->getStatusTag(), $this->a ) );

            $div->children( $line );
        }

        return $wrapper;
    }

	/**
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
    public function formBody() {
	    $rememberMe = $this->a->isExtended ? Extended::rememberMe() : '';
	    $images     = $this->a->isExtended ? Extended::cardLogos() : '';
	    $cardFields = $this->a->isExtended ? Extended::cardInputs() : self::oneLineLayoutInputs();
	    $savedCards = $this->a->isExtended ? Extended::savedCards() : '';
	    $cardholder = $this->a->isExtended ? Extended::cardholderName() : '';

	    $div = parent::formBody();
	    $div->id('adk-method-card')->getClass()->add('adk-method-item');

	    if ( $cardholder ) {
	    	$div->children( $cardholder );
	    }

	    if ( $images ) {
		    $div->children( $images );
	    }

	    if ( $cardFields ) {
		    $div->children( $cardFields );
	    }

	    if ( $rememberMe || $savedCards ) {
		    $div->children( (new Hr())
			    ->style()
			    ->margin()->top('10px')->stop()
			    ->border()->color( PaymentForm::secondaryColor() )->width('1px' )->stop()
			    ->stop());
	    }

	    if ( $rememberMe ) {
		    $div->children( $rememberMe );
	    }

	    if ( $savedCards ) {
		    $div->children( $savedCards );
	    }

	    return $div;
    }

    static public function oneLineLayoutInputs() {
	    return (new Div())->id( 'p-card' )->getClass()->add("combined-wrapper");
    }

	/**
	 * @param $savedCard
	 * @param $cardSecret
	 * @param array $data
	 * @throws \Advertikon\Exception
	 */
    public function savedCardCase( $savedCard, array &$data ) {
    	if ( Advertikon::PARANOID_MODE ) {
		    $customer = \Advertikon\Stripe\Customer::get();
		    $paymentMethod = \Stripe\PaymentMethod::retrieve( $savedCard );

		    if ( !$customer->exists() ) {
			    throw new Exception( 'Customer missing' ); // where the card went from
		    }

		    if ( $customer->stripeId() !== $paymentMethod->customer ) {
			    throw new \Advertikon\Exception( 'Card does not exist' ); // not our card?
		    }
	    }

	    $data['payment_method'] = $savedCard;
    }

	/**
	 * @param $saveCard
	 * @param $cardSecret
	 * @param $data
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
    public function saveCardCase( array &$data = [] ) {
	    $customer = \Advertikon\Stripe\Customer::get();

	    if ( !$customer->exists() ) {
		    if ( $this->a->customer->isLogged() ) {
			    $stripeCustomer = \Advertikon\Stripe\Customer::createNew();
			    \Advertikon\Stripe\Customer::createNewWithStripeCustomer( $stripeCustomer );

		    } else if ( isset( $this->session->data['guest'] ) ) {
			    $stripeCustomer = \Advertikon\Stripe\Customer::createNewGuest();

		    } else {
			    $stripeCustomer = \Advertikon\Stripe\Customer::createNewEmpty();
		    }

		    $data['customer'] = $stripeCustomer->id;
	    }

	    $data['setup_future_usage'] = 'off_session';
    }

	/**
	 * @param array $data
	 * @param bool $saveCard
	 * @return array
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
    public function getIntentData( array $data = [] ) {
        if ( !isset( $this->a->session->data['order_id'] ) ) {
            throw new Exception( 'Order is missing' );
        }

        $orderId = $this->a->session->data['order_id'];
        $orderPrice = new OrderPrice();
	    $stripeCustomer = Customer::get();

        $intent_data = array_merge_recursive( [
            'amount'               => $orderPrice->total( true ),
            'currency'             => strtolower( $orderPrice->currency() ),
            'payment_method_types' => [ 'card' ],
            'metadata'             => $this->a->getMetadata( ['order_id' => $orderId ] ),
            'description'          => $this->a->shortcode()->do_shortcode( Setting::get( 'charge_description', $this->a ) ),
            'statement_descriptor' => Setting::get( 'statement_descriptor', $this->a ) ?: null,
            'capture_method'       => $this->a->doCapturePayment() ? 'automatic' : 'manual'
        ], $data );


        if ( $stripeCustomer->exists() ) {
            $intent_data['customer'] = $stripeCustomer->stripeId();
        }

        if ( Setting::get( 'receipt_email', $this->a ) ) {
            $intent_data['receipt_email'] = $this->a->shortcode()->shortcode_customer_email();
        }

        return $intent_data;
    }
}

class Vendor {
    private $tag;
    private $title;
    private $icon;

    /**
     * @var Advertikon
     */
    private $a;

    /**
     * Vendor constructor.
     * @param null $tag
     * @param null $title
     * @param null $iconPath
     * @throws \Advertikon\Exception
     */
    public function __construct( $tag = null, $title = null, $iconPath = null ) {
        $this->a = Advertikon::instance();

        if ( !is_null( $tag ) ) {
            $this->setTag( $tag );
        }

        if ( !is_null( $title ) ) {
            $this->setTitle( $title);
        }

        if ( !is_null( $iconPath ) ) {
            $this->setIcon( $iconPath );
        }
    }

    /**
     * @return mixed
     */
    public function getTag() {
        return $this->tag;
    }

    /**
     * @param mixed $tag
     */
    public function setTag($tag) {
        $this->tag = $tag;
    }

    /**
     * @return mixed
     */
    public function getTitle() {
        return $this->title;
    }

    /**
     * @param mixed $title
     */
    public function setTitle($title) {
        $this->title = $title;
    }

    /**
     * @return mixed
     */
    public function getIcon() {
        return $this->icon;
    }

    /**
     * @param mixed $icon
     * @throws \Advertikon\Exception
     */
    public function setIcon($icon) {
        $this->icon = $this->a->u()->path_to_url( PaymentOption::$iconFolder . $icon );
    }

    public function isEnabled() {
        return Setting::get( $this->getStatusTag(), $this->a, true );
    }

    public function getStatusTag() {
        return "card_{$this->getTag()}_status";
    }
}