<?php


namespace Advertikon\Stripe\PaymentOption;


use Advertikon\Element\Anchor;
use Advertikon\Element\Bootstrap\Message;
use Advertikon\Element\Button;
use Advertikon\Element\Span;
use Advertikon\Element\Text;
use Advertikon\Setting;
use Advertikon\Stripe\Customer;
use Advertikon\Stripe\Exception;
use Advertikon\Stripe\OrderPrice;
use Advertikon\Stripe\PaymentOption;

class PaymentRequest extends PaymentOption {
    public $tag = PaymentOption::PAYMET_REQUEST;
    public $title = 'Payment&nbsp;Request';
    private $isChrome = false;
    private $isSafari = false;
    private $isIe = false;

    public function __construct() {
        parent::__construct();
        $browscap = $this->a->browscap();

        if ( $browscap ) {
            switch( $browscap->Browser ) {
                case 'Chrome':
                    $this->title = 'Google&nbsp;Pay';
                    $this->isChrome = true;
                    break;
                case 'IE':
                    $this->title = 'Microsoft&nbsp;Pay';
                    $this->isIe = false;
                    break;
                case 'Safari':
                    $this->title = 'Apple&nbsp;Pay';
                    $this->isSafari = false;
                    break;
            }
        }
    }

    private function isSupportedBrowser() {
        $browscap = $this->a->browscap();
        $this->a->log( 'Browser: ' . $browscap->Browser );
        return in_array( $browscap->Browser, [ 'Chrome', 'IE', 'Safari' ] );
    }

    protected function isEnabled() {
        return parent::isEnabled() && $this->isSupportedBrowser();
    }

    function icon() {
        if ( $this->isChrome ) {
            return PaymentOption::$iconFolder . 'google_pay.svg';

        } else if ( $this->isSafari ) {
            return PaymentOption::$iconFolder . 'applepay.svg';

        } else if ( $this->isIe ) {
            return PaymentOption::$iconFolder . 'microsoft_pay.svg';

        } else {
            return PaymentOption::$iconFolder . 'payment_request.svg';
        }
    }

    /**
     * @return \advertikon\element\Div
     * @throws \Exception
     */
    public function getAdminSettings() {
        $wrapper = parent::getAdminSettings();

        $text = new Span( new Text( $this->a->__( 'This option combines Apple Pay, Google Pay, and Microsoft Pay' ) ) );
        $anchor = new Anchor( 'https://stripe.com/docs/stripe-js/elements/payment-request-button',
            $this->a->__( 'More details'), true );

        $text->style()->margin()->right( '5px' );

        $label = new Message( new Span( $text, $anchor) );
        $label->order( -10 );
        $wrapper->children( $label );

        return $wrapper;
    }

    /**
     * @return \Advertikon\Element\Element
     * @throws \Exception
     */
    public function formHeader() {
        $item = parent::formHeader();
        $item->style()->display('none!important' )->cursor('pointer');
        return $item;
    }

    /**
     * @return \Advertikon\Element\Element|void
     * @throws \Exception
     */
    public function formBody() {
        $wrapper = parent::formBody();
        $wrapper->style()->display('none');
        $wrapper->id('payment-request-wrapper');
        $wrapper->children( (new Button())->id('payment-request-button')->style()->display('block')->stop() );

        return $wrapper;
    }

    /**
     * @param array $data
     * @return array
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function getIntentData( array $data = [] ) {
        if ( !isset( $this->a->session->data['order_id'] ) ) {
            throw new Exception( 'Order is missing' );
        }

        $orderId = $this->a->session->data['order_id'];
        $orderPrice = new OrderPrice();
        $stripeCustomer = Customer::get();

        $intent_data = array_merge_recursive( [
            'amount'               => $orderPrice->total( true ),
            'currency'             => $orderPrice->currency(),
            'metadata'             => $this->a->getMetadata( ['order_id' => $orderId, 'integration_check' => 'accept_a_payment' ] ),
            'description'          => $this->a->shortcode()->do_shortcode( Setting::get( 'charge_description', $this->a ) ),
            'statement_descriptor' => Setting::get( 'statement_descriptor', $this->a ) ?: null,
        ], $data );

        if ( $stripeCustomer->exists() ) {
            $intent_data['customer'] = $stripeCustomer->stripeId();
        }

        if ( Setting::get( 'receipt_email', $this->a ) ) {
            $intent_data['receipt_email'] = $this->a->shortcode()->shortcode_customer_email();
        }

        return $intent_data;
    }
}