<?php
/**
 * Advertikon Recurring Order Class
 * @author Advertikon
 * @package Stripe
 * @version 5.0.44  
 */

namespace Advertikon\Stripe;


use Advertikon\Setting;
use Advertikon\Sql;

class Recurring {
	const TABLE = 'advertikon_stripe_recurring';

	const RECURRING_STATUS_SUCCESS = 1;
	const RECURRING_STATUS_FAIL    = 4;

	private $a;
    private $recurringOrderId;
    private $stripeSubscription;
    private $id;
    private $account;

    /**
     * @param array $recurring
     * @return \Stripe\Product|\Stripe\StripeObject
     * @throws \Advertikon\Exception
     * @throws \Stripe\Error\Api
     */
	static function getStripeProduct( array $recurring ) {
		$list = \Stripe\Product::all( ['limit' => 100 ] );
		$a = Advertikon::instance();
		$a->log( 'Getting Stripe product' );

		/** @var \Stripe\Product $product */
		foreach ($list->data as $product ) {
			if ( self::compareProduct( $product, $recurring ) ) {
				$a->log( "Product match found: {$product->name} === {$recurring['recurring']['name']}");
				return $product;
			}
		}

		$a->log( 'Matching Stripe product not found' );
		return self::createStripeProduct( $recurring );
	}

	static private function compareProduct( \Stripe\Product $product, array $recurring ) {
		return $product->name === $recurring['recurring']['name'];
	}

    /**
     * @param array $recurring
     * @return \Stripe\Product
     * @throws \Advertikon\Exception
     */
	static private function createStripeProduct( array $recurring ) {
		Advertikon::instance()->log( 'Creating new Stripe product' );
		return \Stripe\Product::create( [
			'name' => $recurring['recurring']['name'],
			'type' => 'good',
		] );
	}

	/**
	 * @param \Stripe\Price $price
	 * @param array $recurring
	 * @return bool
	 * @throws \Advertikon\Exception
	 * @throws \Exception
	 */
	static private function comparePrice( \Stripe\Price $price, array $recurring ) {
		$a = Advertikon::instance();
		$orderPrice = new OrderPrice();

		$currency = $orderPrice->currency();
		$a->log( "Comparing price {$price->id} with recurring {$recurring['recurring']['name']}" );

		if ( (bool)$price->recurring->trial_period_days !== OrderPrice::isTrial( $recurring['recurring'] ) ) {
			$a->log( sprintf( 'Different trial period: %s', $price->recurring->trial_period_days ) );
			return false;
		}

		if ( OrderPrice::isTrial( $recurring ) && $price->recurring->trial_period_days !== self::getTrialDays( $recurring ) ) {
			$a->log( sprintf( 'Different trial duration: %s != %s', $price->recurring->trial_period_days, self::getTrialDays( $recurring ) ) );
			return false;
		}

		// Currency match
		if ( strtolower( $price->currency ) !== strtolower( $currency ) ) {
			$a->log( sprintf( 'Different currency: %s != %s', $price->currency, $currency ) );
			return false;
		}

		$ocAmount = OrderPrice::price( $recurring['price'], true );

		if( $price->amount !=  $ocAmount ) {
			$a->log( sprintf( 'Different amount: %s != %s', $price->amount, $ocAmount ) );
			return false;
		}

		// Cycles and frequencies match
		if ( strtolower( $recurring['frequency'] ) === 'semi_month' && $price->recurring->interval === 'week' ) {
			if ( ( $recurring['cycle'] * 2 ) !== $price->recurring->interval_count ) {
				$a->log( sprintf( 'Different frequency: %s != %s', $recurring['cycle'] * 2, $price->recurring->interval_count ) );
				return false;
			}

		} elseif ( strtolower( $recurring['frequency'] ) !== strtolower( $price->recurring->interval ) ) {
			$a->log( sprintf( 'Different frequency: %s != %s', $recurring['frequency'], $price->recurring->interval ) );
			return false;

		} elseif ( $recurring['cycle'] != $price->recurring->interval_count ) {
			$a->log( sprintf( 'Different duration: %s != %s', $recurring['cycle'], $price->recurring->interval_count ) );
			return false;
		}

		$a->log( 'Match found' );
		$a->log( $recurring );
		$a->log( $price );

		return true;
	}

	/**
	 * @param array $recurring
	 * @return \Stripe\Price
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 * @throws \Stripe\Exception\ApiErrorException
	 * @throws \Exception
	 */
	static private function createStripePrice( array $recurring, $productId ) {
		$currency = strtolower( (new OrderPrice())->currency() );
		$a = Advertikon::instance();
		$a->log( 'Creating new Stripe price' );

		return \Stripe\Price::create([
			'unit_amount' => OrderPrice::price( $recurring['price'] ),
			'currency'    => strtolower( $currency ),
			'recurring'   => [
				'interval'          => strtolower( $recurring['frequency'] ) === 'semi_month' ?
                    'week' : $recurring['frequency'],

				'interval_count'    => strtolower( $recurring['frequency'] ) === 'semi_month' ?
                    $recurring['cycle'] * 2 : $recurring['cycle'],

				'trial_period_days' => OrderPrice::isTrial( $recurring ) ? self::getTrialDays( $recurring ) : null,
			],
			'product'     => $productId,
			'metadata'    => $a->getMetadata(),
		]);
	}

    /**
     * Returns OpenCart's recurring plan status depend on Stripe's recurring status
     * @param string $status Stripe subscription status
     * @return int
     */
    static public function getSubscriptionStatus( $status ) {
        switch( $status ) {
            case 'trialing' :
                return 6; // Pending
                break;
            case 'active' :
                return version_compare( VERSION, '2.2.0.0', '>=' ) ? 1 : 2; //active
                break;
            case 'unpaid' :
                return version_compare( VERSION, '2.2.0.0', '>=' ) ? 4 : 3; //suspended
                break;
            case 'canceled' :
                return version_compare( VERSION, '2.2.0.0', '>=' ) ? 3 : 4; //canceled
                break;
            case 'past_due' :
            case 'incomplete_expired' :
                return 5; //expired
                break;
            case 'incomplete' :
                return version_compare( VERSION, '2.2.0.0', '>=' ) ? 2 : 1; //inactive
                break;
        }

        return null;
    }

    /**
     * @param $recurringOrderId
     * @return Recurring
     * @throws \Advertikon\Exception
     */
	static public function get( $recurringOrderId ) {
	    $query = Sql::select( self::TABLE, true );
	    $data = $query->where('recurring_order_id')->equal($recurringOrderId)->end()->run();
	    return new self( isset( $data[0] ) ? $data[ 0 ] : []  );
    }

    /**
     * @param $subscriptionId
     * @return Recurring
     * @throws \Advertikon\Exception
     */
    static public function getBySubscriptionId( $subscriptionId ) {
        $query = Sql::select( self::TABLE, true );
        $data = $query->where('subscription_id')->equal($subscriptionId)->end()->run();
        return new self( isset( $data[0] ) ? $data[ 0 ] : []  );
    }

    /**
     * Recurring constructor.
     * @param array $data
     * @throws \Advertikon\Exception
     */
	public function __construct( array $data = [] ) {
		$this->a = Advertikon::instance();

		if ( $data ) {
		    $this->init( $data );
        }
	}

	public function exists() {
	    return !is_null( $this->recurringOrderId ) && !is_null( $this->stripeSubscription );
    }

	public function orderId() {
		return $this->recurringOrderId;
	}

	public function subscriptionId() {
		return $this->stripeSubscription;
	}

    /**
     * @throws \Advertikon\Exception
     */
	public function save() {
	    if ( !is_null( $this->id ) ) {
	        $query = Sql::update( self::TABLE, true );
	        $query->set([
	            'recurring_order_id' => $this->recurringOrderId,
                'subscription_id'    => $this->stripeSubscription,
            ] )
                ->where('id')->equal($this->id)
                ->end()->run();

        } else {
            $query = Sql::insert( self::TABLE, true );
            $query->set( [
                'recurring_order_id' => $this->recurringOrderId,
                'subscription_id'    => $this->stripeSubscription,
	            'account'            => StripeAccount::getCurrentAccount()->getCode()
            ] )->run();

            $this->refresh();
        }
    }

    /**
     * @param array $recurring
     * @param $stripeCustomerId
     * @return \Stripe\Subscription
     * @throws Exception
     * @throws \Advertikon\Exception
     * @throws \Exception
     */
    public function create( array $recurring, $stripeCustomerId ) {
	    $product = self::getStripeProduct( $recurring );
	    $currency = OrderPrice::getCurrency();
        $recurringOrderId = $this->addOcRecurring( $recurring );

        $subscription = \Stripe\Subscription::create( [
            'customer' => $stripeCustomerId,
            'items' => [ [
                'quantity'   => $recurring['quantity'],
                'price_data' => [
                    'unit_amount' => OrderPrice::recurringPrice( $recurring, true ),
                    'currency'    => strtolower( $currency ),
                    'product'     => $product->id,
                    'recurring'   => [
                        'interval'       => strtolower( $recurring['recurring']['frequency'] ) === 'semi_month' ?
                            'week' : $recurring['recurring']['frequency'],

                        'interval_count' => strtolower( $recurring['recurring']['frequency'] ) === 'semi_month' ?
                            $recurring['recurring']['cycle'] * 2 : $recurring['recurring']['cycle'],
                    ],
                ]
            ] ],
            'expand'            => ['latest_invoice.payment_intent'],
            'trial_period_days' => OrderPrice::isTrial( $recurring['recurring'] ) ?
                self::getTrialDays( $recurring ) : null,

            'metadata'          => $this->a->getMetadata( [ 'order_recurring_id' => $recurringOrderId ] ),
            'cancel_at'         => self::getEndDate( $recurring )
        ] );

        $this->stripeSubscription = $subscription->id;
        $this->recurringOrderId = $recurringOrderId;
        $this->save();

        return $subscription;
    }

    /**
     * @throws \Advertikon\Exception
     */
    private function refresh() {
        $data = Sql::select(self::TABLE, true )
            ->where('recurring_order_id')->equal($this->recurringOrderId)
            ->where('subscription_id')->equal($this->stripeSubscription)
            ->end()->run();

        $this->init( isset( $data[0] ) ? $data[0] : [] );
    }

    private function init( array $data ) {
        $this->recurringOrderId   = isset( $data['recurring_order_id'] ) ? $data['recurring_order_id'] : null;
        $this->stripeSubscription = isset( $data['subscription_id'] ) ? $data['subscription_id'] : null;
        $this->id                 = isset( $data['id'] ) ?$data['id'] : null;
        $this->account            = isset( $data['account'] ) ?$data['account'] : null;
    }

    /**
     * @param array $item
     * @return int|\stdClass
     * @throws \Advertikon\Exception
     */
	public function addOcRecurring( array $item ) {
		$this->a->log( 'Placing OC recurring order' );
		$this->a->load->model( 'checkout/recurring' );

        $set = [
            'order_id'              => $this->a->session->data['order_id'],
            'date_added'            => new Sql\RawValue( 'NOW()' ),
            'status'                => self::getSubscriptionStatus( 'inactive' ),
            'product_id'            => $item['product_id'],
            'product_name'          => $item['name'],
            'product_quantity'      => $item['quantity'],
            'recurring_id'          => $item['recurring']['recurring_id'],
            'recurring_name'        => $item['recurring']['name'],
            'recurring_description' => $this->describeSubscription( $item ),
            'recurring_frequency'   => $item['recurring']['frequency'],
            'recurring_cycle'       => $item['recurring']['cycle'],
            'recurring_duration'    => $item['recurring']['duration'],
            'recurring_price'       => $item['recurring']['price'],
            'trial'                 => $item['recurring']['trial'],
            'trial_frequency'       => $item['recurring']['trial_frequency'],
            'trial_cycle'           => $item['recurring']['trial_cycle'],
            'trial_duration'        => $item['recurring']['trial_duration'],
            'trial_price'           => $item['recurring']['trial_price'],
            'reference'             => '',
        ];

		$recurringOrderId = Sql::insert( 'order_recurring', true )->set( $set )->run();
		$this->a->log( sprintf( 'OpenCart recurring order created with ID %s', $recurringOrderId ) );

		return $recurringOrderId;
	}

    /**
     * @param array $r
     * @return int|null
     * @throws \Exception
     */
	static public function getEndDate( array $r ) {
        if ( $r['recurring']['cycle'] * $r['recurring']['duration'] > 0 ) {
            $name = strtolower( $r['recurring']['frequency'] ) === 'semi_month' ?
                'fortnight' : strtolower( $r['recurring']['frequency'] );

            $duration = $r['recurring']['cycle'] * $r['recurring']['duration'];
            $date = new \DateTime("+{$duration} $name" );
            self::addTrialPeriod( $r, $date );
            return $date->getTimestamp();
        }

        return null;
    }

    /**
     * @param $item
     * @return string
     * @throws Exception
     * @throws \Advertikon\Exception
     */
	private function describeSubscription( $item ) {
	    $currency = (new OrderPrice())->currency();

        if ($item['recurring']['trial']) {
            $trial_amt = $this->a->currency->format(
                $this->a->tax->calculate(
                    $item['recurring']['trial_price'],
                    $item['tax_class_id'],
                    $this->a->config->get('config_tax')
                ),
                $currency,
                false,
                false
            ) * $item['quantity'] . ' ' . $this->a->session->data['currency'];

            $trial_text =  $this->a->__( '%s every %s %s for %s payments then ',
                $trial_amt,
                $item['recurring']['trial_cycle'],
                $item['recurring']['trial_frequency'],
                $item['recurring']['trial_duration']
            );

        } else {
            $trial_text = '';
        }

        $recurring_amt = $this->a->currency->format(
            $this->a->tax->calculate($item['recurring']['price'], $item['tax_class_id'], $this->a->config->get('config_tax')),
            $currency,
            false,
            false
        )  * $item['quantity'] . ' ' . $currency;

        $recurring_description = $trial_text . $this->a->__( '%s every %s %s',
            $recurring_amt,
            $item['recurring']['cycle'],
            $item['recurring']['frequency']
        );

        if ($item['recurring']['duration'] > 0) {
            $recurring_description .= $this->a->__( ' for %s payments', $item['recurring']['duration']) ;
        }

        return $recurring_description;
    }

	/**
	 * Updates OC subscription status corresponding to Stripe subscription's status
	 * @param object $subscr Stripe subscription
	 * @throws \Advertikon\Exception on error
	 */
	public function updateStatus( \Stripe\Subscription $subscription ) {
		$this->updateRecurringStatus( $this->getSubscriptionStatus( $subscription->status ) );
		$this->fixOrderStatus();
	}

    /**
     * @param $status
     * @throws \Advertikon\Exception
     */
	private function updateRecurringStatus( $status ) {
        $query = Sql::update( 'order_recurring', true );
        $query->set( [ 'status' => $status ] )->where('order_recurring_id' )->equal($this->recurringOrderId )
            ->end()->run();
    }

    /**
     * Cancels Stripe subscription
     * @throws \Advertikon\Exception
     */
	public function cancel() {
		$account = $this->account ?: StripeAccount::getFallbackCode();

		if ( is_null($account ) ) {
			throw new Exception("Undefined account" );
		}

		$this->a->setAccount( $account );

        $subscription = \Stripe\Subscription::retrieve($this->stripeSubscription);
        $subscription->delete();

        $this->updateRecurringStatus( $this->getSubscriptionStatus( "canceled" ) );
	}

    /**
     * @throws \Advertikon\Exception
     */
	public function cancelAtPeriodEnd() {
		$account = $this->account ?: StripeAccount::getFallbackCode();

		if ( is_null($account ) ) {
			throw new Exception("Undefined account" );
		}

		$this->a->setAccount( $account );

        try {
            \Stripe\Subscription::update($this->stripeSubscription, [ 'cancel_at_period_end' => true ] );

        } catch ( \Exception $e) {
            $this->updateRecurringStatus( $this->getSubscriptionStatus( "canceled" ) );
            $this->a->error( $e );
        }
    }

    /**
     * Returns trial period days quantity
     * @param array $r
     * @return Integer
     * @throws \Exception
     */
	static public function getTrialDays( array $r ) {
		if ( OrderPrice::isTrial( $r['recurring'] ) ) {
			switch( strtolower( $r['recurring']['trial_frequency'] ) ) {
				case 'day' :
					$days = 1;
				break;
				case 'week' :
					$days = 7;
				break;
				case 'month' :
					$today = new \DateTime;
					$month = new \DateTime( '+1 month' );
					$days = $today->diff( $month, true )->format( '%a' );
				break;
				case 'semi_month' :
					$today = new \DateTime;
					$month = new \DateTime( '+1 fortnight' );
					$days = $today->diff( $month, true )->format( '%a' );
				break;
				case 'year' :
					$today = new \DateTime;
					$month = new \DateTime( '+1 year' );
					$days = $today->diff( $month, true )->format( '%a' );
				break;
			}

			$days = $days * $r['recurring']['trial_cycle'] * $r['recurring']['trial_duration'];
		}

		return $days;
	}

    static public function addTrialPeriod( array $r, \DateTime $date ) {
        if ( $r['recurring']['trial'] && $r['recurring']['trial_cycle'] * $r['recurring']['trial_duration'] > 0 ) {
            $frequency = $r['recurring']['trial_frequency'] === 'semi_month' ? 'fortnight' :
                $r['recurring']['trial_frequency'];

            $interval = $r['recurring']['trial_cycle'] * $r['recurring']['trial_duration'];
            $date->add( \DateInterval::createFromDateString( "$interval $frequency" ) );
        }
    }

	/**
	 * Adds subscription transaction line
	 * @param \Stripe\Invoice $invoice
	 * @return bool
	 * @throws \Advertikon\Exception on invoice duplication and on error
	 */
	public function addTransaction( \Stripe\Invoice $invoice ) {
		$amount =  $this->a->currency->convert(
			OrderPrice::fromCents( $invoice->total, $invoice->currency ),
			strtoupper( $invoice->currency ),
			$this->a->config->get( 'config_currency' )
		);

		$status = $invoice->status === "paid" ? Recurring::RECURRING_STATUS_SUCCESS : Recurring::RECURRING_STATUS_FAIL;
		$count = Sql::select('order_recurring_transaction', true )
			->where('reference')->equal($invoice->id)->end()->run();

		if ( count( $count ) > 0 ) {
		    $this->a->log( "Invoice {$invoice->id} has been processed already" );
			return $this->updateInvoiceTransaction( $invoice, $amount, $status );
		}

		return $this->addInvoiceTransaction( $invoice, $amount, $status );
	}

    /**
     * @param \Stripe\Invoice $invoice
     * @param $amount
     * @param $status
     * @return int|\stdClass
     * @throws \Advertikon\Exception
     */
	private function addInvoiceTransaction( \Stripe\Invoice $invoice, $amount, $status ) {
        return Sql::insert('order_recurring_transaction', true )->set([
            'order_recurring_id' => $this->recurringOrderId,
            'reference'          => $invoice->id,
            'amount'             => $amount,
            'type'               => $status,
            'date_added'         => new Sql\RawValue( 'NOW()' )
        ])->run();
    }

    /**
     * @param \Stripe\Invoice $invoice
     * @param $amount
     * @param $status
     * @return mixed
     * @throws \Advertikon\Exception
     */
    private function updateInvoiceTransaction( \Stripe\Invoice $invoice, $amount, $status ) {
        return Sql::update('order_recurring_transaction', true )->set([
            'amount'             => $amount,
            'type'               => $status,
            'date_added'         => new Sql\RawValue( 'NOW()' )
        ])->where('reference' )->equal($invoice->id )->end()->run();
    }

    /**
     * @throws \Advertikon\Exception
     */
    private function fixOrderStatus() {
        $query = Sql::select('order_recurring', true )->field('order_id' )
            ->where('order_recurring_id')->equal($this->recurringOrderId)->end()->run();

        if ( $query ) {
            $ocOrder = \Advertikon\Order::get_by_id( $query[0]['order_id'], $this->a );

            if ( (int)$ocOrder->get_status() === 0 && (float)$ocOrder->get_total() === 0.0 ) {
                $order_model = $this->a->get_order_model( 'catalog' );
                $status = Setting::get( 'status_captured', $this->a );
                $this->a->log( "Changing status of order {$query[0]['order_id']} to $status" );
                $order_model->addOrderHistory( $query[0]['order_id'], $status, '', true, true );

            } else {
                $this->a->log('Do not have to fix order status');
            }

        } else {
            $this->a->error( 'No order corresponding to recurring one' );
        }
    }

    /**
     * @return \Stripe\Subscription|null
     * @throws Exception
     * @throws \Advertikon\Exception
     */
    public function fetch() {
        $account = $this->account ?: StripeAccount::getFallbackCode();

        if ( is_null($account ) ) {
            throw new Exception("Undefined account" );
        }

        $this->a->setAccount( $account );

        try {
            $this->a->setAccount( $account );
            return \Stripe\Subscription::retrieve($this->stripeSubscription );

        } catch ( \Exception $e) {
            // TODO: delete
            $this->a->error( $e );
        }

        return null;
    }
}