<?php
/**
 * Advertikon step.php Class
 * @author Advertikon
 * @package
 * @version 5.0.44  
 */

namespace Advertikon\Stripe;


use Stripe\Exception\ApiErrorException;

class Step {
	/** @var Advertikon */
	private $a;
	private $processedSubscriptions;
	private $oneTimePaid;
	private $failedInvoice;
	private $failedCartId;

	/**
	 * Step constructor.
	 * @throws \Advertikon\Exception
	 */
	public function __construct() {
		$this->a = Advertikon::instance();

		if ( !isset( $this->a->session->data['adk_payment'] ) ) {
			$this->reset();
		}

        $this->a->session->data['adk_payment']['failed_cart'] = null;

		$this->processedSubscriptions = $this->a->session->data['adk_payment']['processed_subscriptions'];
		$this->oneTimePaid = $this->a->session->data['adk_payment']['one_time_paid'];
		$this->failedInvoice = $this->a->session->data['adk_payment']['failed_invoice'];
		$this->failedCartId = $this->a->session->data['adk_payment']['failed_cart_id'];
	}

	private function reset() {
		$this->a->session->data['adk_payment'] = [
			'processed_subscriptions' => [],
			'one_time_paid'           => false,
			'failed_invoice'          => null,
			'failed_cart_id'          => null,
		];
	}

	/**
	 * @return string
	 * @throws Exception
	 * @throws \Advertikon\Exception
	 */
	public function next() {
		if ( $this->getFailedInvoice() ) {
		    if ( $this->cartIdExists( $this->getFailedCartId() ) ) {
                $this->a->log( 'Next step is: retry' );
                return 'retry';
            }

		    $this->a->log( "Cart ID corresponding to failed invoice does not exist" );
		    $this->voidFailedInvoice();
		    return $this->next();

		} else if ( count( array_diff( $this->getCartIds(), $this->processedSubscriptions ) ) > 0 ) {
			$this->a->log( 'Next step is: subscription' );
			return 'subscription';

		} else if ( !$this->oneTimePaid && (new OrderPrice())->total() > 0 ) {
			$this->a->log( 'Next step is: one-time' );
			return 'one-time';

		} else {
			$this->a->log( 'Next step is: end' );
			$this->reset();
			return 'finish';
		}
	}

	public function processedSubscription( $cartId ) {
		$this->a->session->data['adk_payment']['processed_subscriptions'][] = $cartId;
	}

    public function getProcessedSubscriptions() {
        return $this->a->session->data['adk_payment']['processed_subscriptions'];
    }

	public function oneTimePaid( $status = true ) {
		$this->a->session->data['adk_payment']['one_time_paid'] = $status;
	}

	public function failedInvoice( $invoiceId ) {
		$this->a->session->data['adk_payment']['failed_invoice'] = $invoiceId;
	}

	public function getFailedInvoice() {
		return $this->a->session->data['adk_payment']['failed_invoice'];
	}

    public function failedCartId( $cartId ) {
        $this->a->session->data['adk_payment']['failed_cart_id'] = $cartId;
    }

    public function getFailedCartId() {
        return $this->a->session->data['adk_payment']['failed_cart_id'];
    }

    public function isProcessedSubscription( $cartId ) {
        return in_array( $cartId, $this->getProcessedSubscriptions() );
    }

	private function getCartIds() {
		$ret = [];

		foreach( $this->a->cart->getRecurringProducts() as $product ) {
			$ret[] = $product['cart_id'] ?? $product['key'];
		}

		return $ret;
	}

	private function cartIdExists( $cartId ) {
	    return in_array( $cartId, $this->getCartIds() );
    }

    private function voidFailedInvoice() {
	    $this->a->log( "Voiding invoice {$this->getFailedInvoice()}" );

        try {
            $invoice = \Stripe\Invoice::retrieve($this->getFailedInvoice());
            $invoice->voidInvoice();

        } catch ( \Stripe\Error\InvalidRequest $e) {
            $this->a->error( $e );
        }

        $this->failedInvoice(null);
        $this->failedCartId(null);
    }
}