<?php


namespace Advertikon\Stripe;


use Advertikon\Element\Bootstrap\FormGroup;
use Advertikon\Element\Select;
use Advertikon\Setting;

class StripeAccount {
    const DEFAULT_ACCOUNT = "default";
    static private $currentAccount;

    private $a;

    private $account_name = 'Default';
    private $account_currency = '';
    private $live_public_key = '';
    private $live_secret_key = '';
    private $live_signing_key = '';
    private $test_public_key = '';
    private $test_secret_key = '';
    private $test_signing_key = '';
    private $code = self::DEFAULT_ACCOUNT;

    /**
     * @param $code
     * @return StripeAccount
     * @throws \Advertikon\Exception
     */
    public static function byCode( $code ) {
        return new self( $code );
    }

    /**
     * @param $code
     * @param array $data
     * @return StripeAccount
     * @throws \Advertikon\Exception
     */
    public static function byData( $code, array $data ) {
        return new self( $code, $data );
    }

    /**
     * @param null $currencyCode
     * @return StripeAccount
     * @throws \Advertikon\Exception
     */
    public static function byCurrency( $currencyCode = null ) {
        $a = Advertikon::instance();

        if ( !$a->is_admin() ) {
            if ( is_null( $currencyCode ) ) {
                $currencyCode = isset( $a->session->data['currency'] ) ? $a->session->data['currency'] : null;
            }

            if ( is_null( $currencyCode ) ) {
                throw new Exception( 'Failed to detect account by front currency - there is no currency in the session' );
            }

            foreach ( Setting::get('account', $a, array()) as $code => $data ) {
                if ( $currencyCode === $data['account_currency'] ) {
                    return new self( $code, $data );
                }
            }
        }

        return new self( self::DEFAULT_ACCOUNT );
    }

    static public function setCurrentAccount( $accountCode ) {
        self::$currentAccount = $accountCode;
    }

	/**
	 * @return StripeAccount[]
	 * @throws \Advertikon\Exception
	 */
    static public function all() {
    	$ret = [];

	    foreach ( Setting::get('account', Advertikon::instance(), [] ) as $code => $data ) {
	    	$ret[] = new self( $code, $data );
	    }

	    return $ret;
    }

    /**
     * @return array
     * @throws \Advertikon\Exception
     */
    static public function allNames() {
        $ret = [];

        foreach( self::all() as $account ) {
            $ret[$account->getCode()] = $account->getAccountName();
        }

        return $ret;
    }

    /**
     * @return StripeAccount
     * @throws \Advertikon\Exception
     */
    static function getCurrentAccount() {
        $a = Advertikon::instance();
        $accountCode = self::$currentAccount ?: self::DEFAULT_ACCOUNT;

        foreach ((array)Setting::get('account', $a,  array() ) as $code => $data ) {
            if ( $accountCode === $code ) {
                return new self( $code, $data );
            }
        }

        return null;
    }

	/**
	 * @return mixed|null
	 * @throws \Advertikon\Exception
	 */
    static function getFallbackCode() {
    	$all = self::all();

    	if ( count( $all ) === 1 ) {
    		return $all[0]->getCode();
	    }

    	return null;
    }

    /**
     * StripeAccount constructor.
     * @param null $code
     * @param null $data
     * @throws \Advertikon\Exception
     */
    public function __construct( $code = null , $data = null ) {
        $this->a = Advertikon::instance();
        $this->code = $code;
        
        if ( !$data ) {
            $conf = Setting::get('account', $this->a );
            
            if ( $conf && isset( $conf[$code] ) ) {
                $data = $conf[$code];
            }
        }

        if ( $data ) {
            $this->account_name = $data['account_name'];
            $this->account_currency = $data['account_currency'];
            $this->live_public_key = $data['live_public_key'];
            $this->live_secret_key = $data['live_secret_key'];
            $this->live_signing_key = $data['live_signing_key'];
            $this->test_public_key = $data['test_public_key'];
            $this->test_secret_key = $data['test_secret_key'];
            $this->test_signing_key = $data['test_signing_key'];
        }
    }

    /**
     * @return string
     */
    public function getAccountName() {
        return $this->account_name;
    }

    /**
     * @return string
     */
    public function getAccountCurrency() {
        return $this->account_currency;
    }

    public function secretKey() {
        return Setting::get( 'test_mode', $this->a, false ) ?
            $this->getTestSecretKey() : $this->getLiveSecretKey();
    }

    public function publicKey() {
        return Setting::get( 'test_mode', $this->a, false ) ?
            $this->getTestPublicKey() : $this->getLivePublicKey();
    }

    /**
     * @return string
     */
    public function getLivePublicKey() {
        return $this->live_public_key;
    }

    /**
     * @return string
     */
    public function getLiveSecretKey() {
        return $this->live_secret_key;
    }

    /**
     * @return string
     */
    public function getLiveSigningKey() {
        return $this->live_signing_key;
    }

    /**
     * @return string
     */
    public function getTestPublicKey() {
        return $this->test_public_key;
    }

    /**
     * @return string
     */
    public function getTestSecretKey() {
        return $this->test_secret_key;
    }

    /**
     * @return string
     */
    public function getTestSigningKey() {
        return $this->test_signing_key;
    }

    /**
     * @return string|null
     */
    public function getCode() {
        return $this->code;
    }

    /**
     * Get stripe account template for administrative area
     * @return String
     * @throws \Advertikon\Exception
     */
    public function fields( $errors ) {
        $templ = false;

        if ( is_null( $this->code ) ) {
            $this->code = '{template_name}';
            $templ = true;
        }

        $data['default'] = $this->code === 'default';
        $option = new \Advertikon\Stripe\Option();
        $name_prefix = "account-" . $this->code;

        //Account name field
        $field = 'account_name';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data['account_name'] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Account name' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control account-name',
                'placeholder' => $this->a->__( 'Account name' ),
                'value'       => $this->account_name,
                'custom_data' => [ 'data-name' => \Advertikon\Setting::prefix_name( $name, $this->a ), ],
            ) ),
            'error'   => isset( $errors['account'][ $this->code ]['account_name'] ) ?
                $errors['account'][ $this->code ]['account_name'] : null,
        ) );

        // Account currency field
        $field = 'account_currency';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data['account_currency'] = (new FormGroup())
            ->label( $this->a->__( 'Account currency' ) )
            ->tooltip( $this->a->__( 'Front end currency, associated with this Stripe account' ) )
            ->error( $data, "account/{$this->code}/account_currency" )
            ->element((new Select())
                ->id( $id )
                ->getClass()->add( 'form-control account-currency' )
                ->option( $option->currency_code() )
                ->attributes()->set([
                    'data-name' => \Advertikon\Setting::prefix_name( $name, $this->a ),
                    'data-value' => $this->account_currency
                ])
                ->value( $this->account_currency )
                ->name( $templ ? '' : $name, $this->a )
            );

            // Test secret key field
        $field = 'test_secret_key';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data['test_secret_key'] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Test secret key' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control secret-test secret-secret',
                'placeholder' => $this->a->__( 'Test secret key' ),
                'value'       => $this->test_secret_key,
                'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
            ) ),
            'error'   => isset( $errors['account'][ $this->code ]['test_secret_key'] ) ?
                $errors['account'][ $this->code ]['test_secret_key'] : null,
        ) );

        // Test public key field
        $field = 'test_public_key';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data['test_public_key'] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Test publishable key' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control secret-test secret-public',
                'placeholder' => $this->a->__( 'Test publishable key' ),
                'value'       => $this->test_public_key,
                'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
            ) ),
            'error'   => isset( $errors['account'][ $this->code ]['test_public_key'] ) ?
                $errors['account'][ $this->code ]['test_public_key'] : null,
        ) );

        // Live secret key field
        $field = 'live_secret_key';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id , '-');

        $data['live_secret_key'] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Live secret key' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control secret-live secret-secret',
                'placeholder' => $this->a->__( 'Live secret key' ),
                'value'       => $this->live_secret_key,
                'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
            ) ),
            'error'   => isset( $errors['account'][ $this->code ]['live_secret_key'] ) ?
                $errors['account'][ $this->code ]['live_secret_key'] : null,
        ) );

        // Live public key field
        $field = 'live_public_key';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data['live_public_key'] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Live publishable key' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control secret-live secret-public',
                'placeholder' => $this->a->__( 'Live publishable key' ),
                'value'       => $this->live_public_key,
                'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
            ) ),
            'error'   => isset( $errors['account'][ $this->code ]['live_public_key'] ) ?
                $errors['account'][ $this->code ]['live_public_key'] : null,
        ) );

        // Live signing key
        $field = 'live_signing_key';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data['live_signing_key'] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Live signing secret' ),
            'tooltip'   => $this->a->__( 'Webhook signing secret for live mode. Can be found at Developers > Webhooks section of Stripe Dashboard when viewing live data.' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control secret-signing secret-live',
                'placeholder' => $this->a->__( 'Live signing secret' ),
                'value'       => $this->live_signing_key,
                'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
            ) ),
            'error'   => isset( $errors['account'][ $this->code ]['live_signing_key'] ) ?
                $errors['account'][ $this->code ]['live_signing_key'] : null,
        ) );

        // Public signing key
        $field = 'test_signing_key';
        $id = $name_prefix . '-' . $field;
        $name = $this->a->build_name( $id, '-' );

        $data[ $field ] = $this->a->r()->render_form_group( array(
            'label'   => $this->a->__( 'Test signing secret' ),
            'tooltip'   => $this->a->__( 'Webhook signing secret for test mode. Can be found at Developers > Webhooks section of Stripe Dashboard when viewing test data' ),
            'element' => $this->a->r( array(
                'type'        => 'text',
                'id'          => $id,
                'name'        => $templ ? '' : $name,
                'class'       => 'form-control secret-signing secret-test',
                'placeholder' => $this->a->__( 'Test signing secret' ),
                'value'       => $this->test_signing_key,
                'custom_data' => 'data-name="' . \Advertikon\Setting::prefix_name( $name, $this->a ) . '"',
            ) ),
            'error'   => isset( $errors['account'][ $this->code ][ $field ] ) ?
                $errors['account'][ $this->code ][ $field ] : null,
        ) );

        $data['a'] = $this->a;

        return $this->a->load->view(
            $this->a->get_template( $this->a->type . '/advertikon/stripe/account' ), $data
        );
    }

}